<?php

namespace Drupal\Tests\advanced_mega_menu\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\Url;
use Drupal\Core\Template\Attribute;

/**
 * Tests the recursive macro logic in menu--advanced-mega-menu.html.twig.
 *
 * This test validates the macro's ability to render nested submenus and
 * switch to mega-panel mode while satisfying module-level dependencies.
 *
 * @group advanced_mega_menu
 */
class AdvancedMegaMenuMacroTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'advanced_mega_menu',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    // Disable Twig cache to prevent vfs errors in kernel tests.
    $this->container->get('twig')->setCache(FALSE);

    $this->container->get('theme_installer')->install(['stark']);
    $this->container->get('config.factory')
      ->getEditable('system.theme')
      ->set('default', 'stark')
      ->save();
  }

  /**
   * Helper to create a menu item with all required keys.
   *
   * Why: Prevents warnings and ensures .addClass() methods work correctly.
   */
  protected function createMockItem(string $title, Url $url, array $overrides = []): array {
    return array_merge([
      'title' => $title,
      'url' => $url,
      'attributes' => new Attribute(),
      'mega_menu_rows' => NULL,
      'below' => [],
    // Satisfies expanded check in preprocess.
      'is_expanded' => FALSE,
    ], $overrides);
  }

  /**
   * Helper to render the menu template.
   *
   * How: We add '#menu_name' here to satisfy the very first line of your
   * preprocess function: $menu_name = $variables['menu_name'];.
   */
  protected function renderMenuTemplate(array $items, $icon = ''): string {
    $build = [
      '#theme' => 'menu__advanced_mega_menu',
      '#items' => $items,
      '#submenu_icon' => $icon,
      '#attributes' => new Attribute(),
    // Crucial: Fixes the "Undefined array key menu_name" warning.
      '#menu_name' => 'main',
    ];

    $renderer = $this->container->get('renderer');
    return (string) $renderer->executeInRenderContext(new RenderContext(), function () use ($renderer, $build) {
      return $renderer->render($build);
    });
  }

  /**
   * Tests standard recursion.
   */
  public function testStandardRecursion(): void {
    $url = Url::fromUri('internal:/');
    $items = [
      'parent' => $this->createMockItem('Parent Link', $url, [
        'below' => [
          'child' => $this->createMockItem('Child Link', $url),
        ],
      ]),
    ];

    $output = $this->renderMenuTemplate($items, '<span class="test-icon">▼</span>');

    // Logic: has_children is TRUE because 'below' is not empty.
    $this->assertStringContainsString('class="advanced-mega-menu-item has-submenu"', $output);
    $this->assertStringContainsString('test-icon', $output);
    $this->assertStringContainsString('class="advanced-mega-menu submenu"', $output);
  }

  /**
   * Tests Mega Menu Panel rendering and class order.
   */
  public function testMegaMenuPanelLogic(): void {
    $url = Url::fromUri('internal:/');
    $items = [
      'mega_item' => $this->createMockItem('Mega Link', $url, [
        'mega_menu_rows' => [
          '#has_children' => TRUE,
          '#markup' => '<div class="mega-panel-content">Custom Content</div>',
        ],
      ]),
    ];

    $output = $this->renderMenuTemplate($items);

    // Logic check:
    // 1. addClass('advanced-mega-menu-item')
    // 2. has_children is TRUE (mega_menu_rows['#has_children'] is
    // true) -> 'has-submenu'
    // 3. is_mega is TRUE (mega_menu_rows is defined) -> 'mega'
    // Output must be: "advanced-mega-menu-item has-submenu
    // mega".
    $this->assertStringContainsString('class="advanced-mega-menu-item has-submenu mega"', $output);
    $this->assertStringContainsString('advanced-mega-menu-panel', $output);
    $this->assertStringContainsString('mega-panel-content', $output);
  }

  /**
   * Tests items without children.
   */
  public function testLeafNode(): void {
    $url = Url::fromUri('internal:/');
    $items = ['leaf' => $this->createMockItem('Simple Link', $url)];

    $output = $this->renderMenuTemplate($items, '<i>icon</i>');

    $this->assertStringNotContainsString('has-submenu', $output);
    $this->assertStringNotContainsString('<i>icon</i>', $output);
  }

}
