<?php

namespace Drupal\Tests\advanced_mega_menu\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\Url;

/**
 * Tests the rendering logic of the Advanced Mega Menu Twig template.
 *
 * This test ensures that the template correctly handles the positioning of
 * menu links relative to injected mega menu content, validates CSS class
 * injections, and verifies nested recursive link structures.
 *
 * @group advanced_mega_menu
 */
class AdvancedMegaMenuRenderingTest extends KernelTestBase {

  /**
   * Modules to enable for this test.
   *
   * @var array
   */
  protected static $modules = [
    'system',
    'user',
    'advanced_mega_menu',
  ];

  /**
   * Sets up the test environment.
   *
   * Why:
   * 1. Twig caching is disabled because the Virtual File System (vfs://) used
   * in Kernel tests often fails when Twig tries to create nested directory
   * structures for compiled PHP files.
   * 2. A default theme (Stark) is installed to provide a stable rendering
   * engine.
   */
  protected function setUp(): void {
    parent::setUp();

    // Disable Twig cache to prevent "vfs://.../twig" directory creation errors.
    $this->container->get('twig')->setCache(FALSE);

    // Install 'stark' theme to ensure the ThemeManager has a theme to work
    // with.
    $this->container->get('theme_installer')->install(['stark']);
    $this->container->get('config.factory')
      ->getEditable('system.theme')
      ->set('default', 'stark')
      ->save();
  }

  /**
   * Renders a render array into an HTML string.
   *
   * @param array $render_array
   *   The render array to process.
   *
   * @return string
   *   The rendered HTML output.
   *
   *   How it works:
   *   It uses the 'renderer' service and wraps the call in a RenderContext.
   *   This is required in Kernel tests to prevent "leaked cache metadata"
   *   exceptions, which occur when a bubbleable cacheable object is rendered
   *   outside of a defined context.
   */
  protected function renderMegaMenu(array $render_array): string {
    $renderer = $this->container->get('renderer');
    return (string) $renderer->executeInRenderContext(new RenderContext(), function () use ($renderer, $render_array) {
      return $renderer->render($render_array);
    });
  }

  /**
   * Tests the 'top' positioning logic and dynamic CSS class assignment.
   *
   * Why: Validates that the Twig template respects the 'menu_items_position'
   * setting.
   *
   * How: 1. Uses Url::fromUri('internal:/') because Twig's link() function
   * requires a valid Url object, and plain strings like '/' are invalid in
   * Kernel tests. 2. Compares the string position (strpos) of the menu title
   * vs the mega content.
   */
  public function testTemplateLogic(): void {
    $url = Url::fromUri('internal:/');

    $render_array = [
      '#theme' => 'advanced_mega_menu',
      '#menu_link_contents' => [
        [
          'parent' => [
            'title' => 'Parent Link',
            'url' => $url,
            'attributes' => [],
          ],
        ],
      ],
      '#mega_menu_contents' => [
        [
          'column_layout_mode' => 'custom-layout',
          'columns' => [['<p>Mega Content</p>']],
        ],
      ],
      '#display_settings' => [
        'menu_items_position' => 'top',
        'row_class' => 'my-custom-row',
      ],
    ];

    $output = $this->renderMegaMenu($render_array);

    // Verify CSS injection works for custom classes.
    $this->assertStringContainsString('my-custom-row', $output);

    // Verify positioning: Parent Link should occur before Mega
    // Content in the string.
    $pos_menu = strpos($output, 'Parent Link');
    $pos_mega = strpos($output, 'Mega Content');
    $this->assertLessThan($pos_mega, $pos_menu, 'Menu links should appear ABOVE mega content.');
  }

  /**
   * Tests the 'bottom' positioning logic.
   *
   * Why:
   * Ensures that the default template behavior (or explicit 'bottom' setting)
   * correctly appends the menu links after the mega menu block rows.
   */
  public function testBottomPositioning(): void {
    $url = Url::fromUri('internal:/');

    $render_array = [
      '#theme' => 'advanced_mega_menu',
      '#menu_link_contents' => [
        ['parent' => ['title' => 'Parent Link', 'url' => $url, 'attributes' => []]],
      ],
      '#mega_menu_contents' => [
        ['column_layout_mode' => 'row', 'columns' => [['<p>Mega Content</p>']]],
      ],
      '#display_settings' => [
        'menu_items_position' => 'bottom',
      ],
    ];

    $output = $this->renderMegaMenu($render_array);

    $pos_menu = strpos($output, 'Parent Link');
    $pos_mega = strpos($output, 'Mega Content');

    // Menu should appear AFTER mega content.
    $this->assertGreaterThan($pos_mega, $pos_menu, 'Menu links should appear BELOW mega content.');
  }

}
