<?php

namespace Drupal\Tests\advanced_mega_menu\Kernel;

use Drupal\block_content\Entity\BlockContent;
use Drupal\block_content\Entity\BlockContentType;
use Drupal\KernelTests\KernelTestBase;
use Drupal\block\Entity\Block;
use Drupal\Core\Entity\Entity\EntityViewDisplay;

/**
 * Tests the MegaMenuBlockRenderer service.
 *
 * @group advanced_mega_menu
 */
class MegaMenuBlockRendererTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'filter',
    'block',
    'block_content',
    'views',
    'advanced_mega_menu',
  ];

  /**
   * The service under test.
   *
   * @var \Drupal\advanced_mega_menu\Service\MegaMenuBlockRenderer
   */
  protected $renderer;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('block_content');
    $this->installConfig(['system', 'field', 'text', 'filter', 'views', 'block_content']);

    // REQUIRED: Install a theme to allow the ViewBuilder to function.
    $this->container->get('theme_installer')->install(['stark']);
    $this->container->get('config.factory')->getEditable('system.theme')->set('default', 'stark')->save();

    $this->renderer = $this->container->get('advanced_mega_menu.block_renderer');
  }

  /**
   * Tests rendering a Content Block (BlockContent entity).
   */
  public function testRenderContentBlock(): void {
    // 1. Setup: Create a block content type.
    $type = BlockContentType::create([
      'id' => 'basic',
      'label' => 'Basic',
    ]);
    $type->save();

    // 2. REQUIRED: Create an EntityViewDisplay so the ViewBuilder knows
    // what to render.
    EntityViewDisplay::create([
      'targetEntityType' => 'block_content',
      'bundle' => 'basic',
      'mode' => 'full',
      'status' => TRUE,
    ])->save();

    // 3. Create the entity.
    $block_content = BlockContent::create([
      'type' => 'basic',
      'info' => 'My Custom Block',
      'body' => [
        'value' => 'Block body content',
        'format' => 'plain_text',
      ],
    ]);
    $block_content->save();

    // 4. Test rendering.
    $build = $this->renderer->renderContentBlock($block_content->id(), FALSE);

    // Instead of checking #theme (which can vary by view mode),
    // check for the presence of the entity in the render array.
    $this->assertArrayHasKey('#block_content', $build, 'The render array should contain the block content entity.');
    $this->assertEquals($block_content->id(), $build['#block_content']->id());
  }

  /**
   * Tests rendering a Theme Block.
   */
  public function testRenderThemeBlock(): void {
    $block = Block::create([
      'id' => 'test_theme_block',
      'plugin' => 'system_powered_by_block',
      'theme' => 'stark',
      'settings' => ['label' => 'Powered by Drupal'],
    ]);
    $block->save();

    $build = $this->renderer->renderThemeBlock('test_theme_block', TRUE);

    $this->assertEquals('container', $build['#type']);
    $this->assertStringContainsString('Powered by Drupal', (string) $build['label']['#markup']);
    $this->assertArrayHasKey('block', $build);
  }

  /**
   * Tests rendering a View display fallback.
   */
  public function testRenderViewBlockFallback(): void {
    $config = [
      'view_id' => 'non_existent_view',
      'display_id' => 'block_1',
    ];

    $build = $this->renderer->renderViewBlock($config, FALSE);
    $this->assertEquals('Unable to render this view.', (string) $build['#markup']);
  }

}
