<?php

namespace Drupal\Tests\advanced_mega_menu\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\advanced_mega_menu\Entity\MegaMenuContent;
use Drupal\advanced_mega_menu\Form\MegaMenuContentDeleteForm;
use Drupal\Core\Form\FormState;

/**
 * Tests the MegaMenuContent delete confirmation form.
 *
 * @group advanced_mega_menu
 */
class MegaMenuContentDeleteFormTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'advanced_mega_menu',
  ];

  /**
   * The entity under test.
   *
   * @var \Drupal\advanced_mega_menu\Entity\MegaMenuContent
   */
  protected $entity;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install system config and module config.
    $this->installConfig(['system', 'advanced_mega_menu']);

    // Create a config entity using 'form_id' as the primary ID key.
    $this->entity = MegaMenuContent::create([
      'form_id' => 'main_nav_mega',
      'label' => 'Main Navigation Mega Menu',
      'menu_id' => 'main',
      'status' => TRUE,
    ]);
    $this->entity->save();
  }

  /**
   * Tests the form structure and display.
   */
  public function testFormDisplay(): void {
    $form_object = MegaMenuContentDeleteForm::create($this->container);
    $form_state = new FormState();

    // buildForm expects the entity ID as the 3rd argument
    // (mimicking route param).
    $form_object->buildForm([], $form_state, $this->entity->id());

    // Verify the question text (stripping tags due to %label placeholder).
    $this->assertStringContainsString(
      'Are you sure you want to delete the Mega Menu Content for Main Navigation Mega Menu?',
      strip_tags((string) $form_object->getQuestion())
    );

    // Verify Cancel URL.
    $this->assertEquals(
      'advanced_mega_menu.megamenu_content.collection',
      $form_object->getCancelUrl()->getRouteName()
    );
  }

  /**
   * Tests the submission and actual deletion from config storage.
   */
  public function testFormSubmission(): void {
    $form_object = MegaMenuContentDeleteForm::create($this->container);
    $form_state = new FormState();

    // Build the form to load the entity into the object.
    $form_object->buildForm([], $form_state, $this->entity->id());

    // Submit the form.
    $form = [];
    $form_object->submitForm($form, $form_state);

    // 1. Verify the entity is deleted from config storage.
    $storage = $this->container->get('entity_type.manager')->getStorage('megamenu_content');
    $reloaded_entity = $storage->load('main_nav_mega');
    $this->assertNull($reloaded_entity, 'Entity was successfully deleted from configuration.');

    // 2. Verify redirect.
    $redirect = $form_state->getRedirect();
    $this->assertEquals('advanced_mega_menu.megamenu_content.collection', $redirect->getRouteName());

    // 3. Verify the success message.
    // We use strip_tags because Drupal wraps %label in
    // <em class="placeholder">.
    $messages = \Drupal::messenger()->all();
    $this->assertNotEmpty($messages['status'], 'A status message was set.');

    $actual_message = strip_tags((string) $messages['status'][0]);
    $expected_message = 'Mega Menu Content for Main Navigation Mega Menu has been deleted.';

    $this->assertEquals($expected_message, $actual_message);
  }

}
