<?php

namespace Drupal\Tests\advanced_mega_menu\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\advanced_mega_menu\Form\MegaMenuLayoutBuilderForm;
use Drupal\Core\Form\FormState;
use Drupal\block_content\Entity\BlockContent;
use Drupal\block_content\Entity\BlockContentType;

/**
 * Tests the logic of MegaMenuLayoutBuilderForm.
 *
 * @group advanced_mega_menu
 */
class MegaMenuLayoutBuilderFormTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'filter',
    'block',
    'block_content',
    'views',
    'advanced_mega_menu',
  ];

  /**
   * The form object under test.
   *
   * @var \Drupal\advanced_mega_menu\Form\MegaMenuLayoutBuilderForm
   */
  protected $formObject;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('block_content');

    // Ensure custom entity schema is installed if it exists.
    if ($this->container->get('entity_type.manager')->hasDefinition('megamenu_content')) {
      $this->installEntitySchema('megamenu_content');
    }

    $this->installConfig(['system', 'field', 'filter', 'block_content', 'advanced_mega_menu']);

    // Initialize the theme system (Required for AJAX rendering).
    $this->container->get('theme_installer')->install(['stark']);
    $this->container->get('config.factory')
      ->getEditable('system.theme')
      ->set('default', 'stark')
      ->save();

    // THE FIX: Disable Twig cache to prevent vfs:// write errors.
    // This tells Twig to compile templates in memory without writing to disk.
    $twig_service = $this->container->get('twig');
    $twig_service->setCache(FALSE);

    $this->formObject = MegaMenuLayoutBuilderForm::create($this->container);
  }

  /**
   * Tests row and column addition logic.
   */
  public function testLayoutModification(): void {
    $form_state = new FormState();
    $form_state->set('layout', []);

    // Test Adding a Row.
    $this->formObject->addRow([], $form_state);
    $layout = $form_state->get('layout');
    $this->assertCount(1, $layout);
    $this->assertTrue($form_state->isRebuilding());

    // Test Adding a Column.
    $form_state->setTriggeringElement(['#name' => 'add_col_0']);
    $this->formObject->addColumn([], $form_state);
    $layout = $form_state->get('layout');
    $this->assertCount(1, $layout[0]['columns']);

    // Test Removing the Row.
    $form_state->setTriggeringElement(['#name' => 'rm_row_0']);
    $this->formObject->removeRow([], $form_state);
    $this->assertCount(0, $form_state->get('layout'));
  }

  /**
   * Tests adding and removing blocks within a column.
   */
  public function testBlockModification(): void {
    $form_state = new FormState();
    $form_state->set('layout', [
      0 => [
        'columns' => [
          0 => ['blocks' => []],
        ],
      ],
    ]);

    // Test Adding a Block.
    $form_state->setTriggeringElement(['#name' => 'add_blk_0_0']);
    $this->formObject->addBlock([], $form_state);
    $layout = $form_state->get('layout');
    $this->assertCount(1, $layout[0]['columns'][0]['blocks']);

    // Test Removing a Block.
    $form_state->setTriggeringElement(['#name' => 'rm_blk_0_0_0']);
    $this->formObject->removeBlock([], $form_state);
    $this->assertCount(0, $form_state->get('layout')[0]['columns'][0]['blocks']);
  }

  /**
   * Tests that block content options are correctly fetched.
   */
  public function testPrepareSourceOptions(): void {
    $type = BlockContentType::create(['id' => 'basic', 'label' => 'Basic']);
    $type->save();

    $block = BlockContent::create(['type' => 'basic', 'info' => 'Test Block']);
    $block->save();

    $form_state = new FormState();
    $method = new \ReflectionMethod(MegaMenuLayoutBuilderForm::class, 'prepareSourceOptions');
    $method->setAccessible(TRUE);
    $method->invoke($this->formObject, $form_state);

    $options = $form_state->get('opt_content');
    $this->assertArrayHasKey('Basic', $options);
    $this->assertEquals('Test Block', $options['Basic'][$block->id()]);
  }

  /**
   * Tests the AJAX refresh callback.
   */
  public function testRefreshAjax(): void {
    $form = [
      'rows_wrapper' => [
        'canvas' => [
          '#type' => 'container',
          '#attributes' => ['id' => 'mega-menu-canvas'],
          '#markup' => 'Canvas Content',
        ],
      ],
    ];
    $form_state = new FormState();

    // Test Case: Simulate trigger for adding a row.
    $form_state->setTriggeringElement(['#name' => 'add_row_op']);

    $response = $this->formObject->refreshAjax($form, $form_state);
    $commands = $response->getCommands();

    // Verify ReplaceCommand.
    $this->assertEquals('insert', $commands[0]['command']);
    $this->assertEquals('#mega-menu-canvas', $commands[0]['selector']);

    // Verify Custom ScrollToCommand.
    $this->assertEquals('megaMenuScrollTo', $commands[1]['command']);
  }

}
