<?php

namespace Drupal\Tests\advanced_mega_menu\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\advanced_mega_menu\Plugin\rest\resource\MegaMenuResource;
use Drupal\advanced_mega_menu\Service\MegaMenuRowBuilder;
use Drupal\rest\ResourceResponse;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Tests the MegaMenuResource REST plugin.
 *
 * @group advanced_mega_menu
 */
class MegaMenuResourceTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'rest',
    'serialization',
    'advanced_mega_menu',
  ];

  /**
   * The mock row builder service.
   *
   * @var \PHPUnit\Framework\MockObject\MockObject
   */
  protected $rowBuilder;

  /**
   * The REST resource under test.
   *
   * @var \Drupal\advanced_mega_menu\Plugin\rest\resource\MegaMenuResource
   */
  protected $resource;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['rest']);

    // Mock the MegaMenuRowBuilder.
    $this->rowBuilder = $this->createMock(MegaMenuRowBuilder::class);

    // Instantiate the resource with mocked dependencies.
    $this->resource = new MegaMenuResource(
      [],
      'advanced_mega_menu_resource',
      [],
      ['json'],
      $this->container->get('logger.factory')->get('advanced_mega_menu'),
      $this->rowBuilder,
      $this->container->get('renderer')
    );
  }

  /**
   * Tests a successful GET request.
   */
  public function testGetSuccess(): void {
    $menu_id = 'main';
    $plugin_id = 'menu_link_content:123';

    // Mock buildMenuItemMegaRows to return a simple render array.
    $render_array = [
      '#markup' => '<div class="mega-menu">Test Content</div>',
    ];

    $this->rowBuilder->expects($this->once())
      ->method('buildMenuItemMegaRows')
      ->with($menu_id, $plugin_id)
      ->willReturn($render_array);

    // Call the GET method.
    $response = $this->resource->get($menu_id, $plugin_id);

    // Assertions.
    $this->assertInstanceOf(ResourceResponse::class, $response);
    $data = $response->getResponseData();

    // Check that HTML was rendered.
    $this->assertStringContainsString('Test Content', (string) $data['megamenu']);

    // Check Cache Metadata.
    $cache_tags = $response->getCacheableMetadata()->getCacheTags();
    $this->assertContains('mega_menu_content:123', $cache_tags);
    $this->assertContains('config:system.menu.main', $cache_tags);

    $this->assertContains('url.path', $response->getCacheableMetadata()->getCacheContexts());
  }

  /**
   * Tests that the resource throws a 404 when no content is found.
   */
  public function testGetNotFound(): void {
    $this->rowBuilder->method('buildMenuItemMegaRows')
      ->willReturn([]);

    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('Mega menu not found or disabled.');

    $this->resource->get('invalid', 'invalid:id');
  }

}
