<?php

namespace Drupal\Tests\advanced_mega_menu\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\advanced_mega_menu\Entity\MegaMenuContent;

/**
 * Tests the MegaMenuRowBuilder service.
 *
 * @group advanced_mega_menu
 */
class MegaMenuRowBuilderTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'advanced_mega_menu',
    'block',
    'block_content',
    'views',
  ];

  /**
   * The builder service.
   *
   * @var \Drupal\advanced_mega_menu\Service\MegaMenuRowBuilder
   */
  protected $rowBuilder;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['advanced_mega_menu']);

    // Ensure global functions in .module are available.
    $this->container->get('module_handler')->loadInclude('advanced_mega_menu', 'module');

    $this->rowBuilder = $this->container->get('advanced_mega_menu.mega_menu_row_builder');
  }

  /**
   * Tests building mega menu rows when the entity exists and is enabled.
   */
  public function testBuildMenuItemMegaRowsSuccess(): void {
    $menu_name = 'main';
    $plugin_id = 'menu_link_content:123-uuid';
    $clean_id = '123-uuid';

    // 1. Create the config entity with enough data to trigger
    // #has_children = TRUE.
    $entity = MegaMenuContent::create([
      'form_id' => $clean_id,
      'menu_id' => $menu_name,
      'status' => TRUE,
      'label' => 'Test Mega Menu',
      'display_settings' => [
        'menu_display_mode' => 'mega_only',
        'row_class' => 'test-row-class',
      ],
      // Provide a mock row structure so getRowsData() isn't empty.
      'rows' => [
        [
          'column_layout_mode' => 'equal',
          'columns' => [
            [
              'blocks' => [
                [
                  'category' => 'theme_id',
                  'theme_id' => 'system_powered_by_block',
                ],
              ],
            ],
          ],
        ],
      ],
    ]);
    $entity->save();

    // 2. Call the builder.
    $build = $this->rowBuilder->buildMenuItemMegaRows($menu_name, $plugin_id);

    // 3. Assertions.
    $this->assertNotNull($build);
    $this->assertEquals('advanced_mega_menu', $build['#theme']);

    // Check that children exist (since we provided rows).
    $this->assertTrue($build['#has_children'], 'The mega menu should report having children.');
    $this->assertNotEmpty($build['#mega_menu_contents']);
  }

  /**
   * Tests that the builder returns NULL if the entity is disabled.
   */
  public function testBuildDisabledMegaMenu(): void {
    $clean_id = 'disabled-id';
    MegaMenuContent::create([
      'form_id' => $clean_id,
      'menu_id' => 'main',
      'status' => FALSE,
    ])->save();

    $build = $this->rowBuilder->buildMenuItemMegaRows('main', "menu_link_content:$clean_id");
    $this->assertNull($build, 'Should return NULL for disabled mega menus.');
  }

  /**
   * Tests the 'menu_and_mega' display mode and item modification.
   */
  public function testMenuAndMegaDisplayMode(): void {
    $clean_id = 'combined-id';
    MegaMenuContent::create([
      'form_id' => $clean_id,
      'menu_id' => 'main',
      'status' => TRUE,
      'display_settings' => ['menu_display_mode' => 'menu_and_mega'],
    ])->save();

    // Mock an item with children in 'below'.
    // In a real scenario, this would be a menu tree array.
    $item = [
      'below' => [
        'child_1' => [
          'title' => 'Child Link',
          'url' => '/',
          'attributes' => [],
        ],
      ],
    ];

    $build = $this->rowBuilder->buildMenuItemMegaRows('main', $clean_id, $item);

    // 1. Verify 'below' was cleared by the builder to prevent duplicates.
    $this->assertEmpty($item['below'], 'The "below" array should be emptied by the builder.');

    // 2. Verify children exist.
    $this->assertTrue($build['#has_children']);
  }

}
