<?php

namespace Drupal\Tests\advanced_mega_menu\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\advanced_mega_menu\Menu\MenuLinkHelper;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\system\Entity\Menu;

/**
 * Tests the MenuLinkHelper service.
 *
 * @group advanced_mega_menu
 */
class MenuLinkHelperTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'menu_link_content',
    'link',
    'advanced_mega_menu',
  ];

  /**
   * The service under test.
   *
   * @var \Drupal\advanced_mega_menu\Menu\MenuLinkHelper
   */
  protected $menuLinkHelper;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('menu_link_content');
    $this->installConfig(['system']);

    $this->menuLinkHelper = new MenuLinkHelper(
      $this->container->get('menu.link_tree')
    );
  }

  /**
   * Tests the menu link flattening and title retrieval.
   */
  public function testGetMenuLinksWithTitles(): void {
    // 1. Create a custom menu.
    $menu_name = 'test-mega-menu';
    Menu::create([
      'id' => $menu_name,
      'label' => 'Test Mega Menu',
    ])->save();

    // 2. Create a parent menu link.
    $parent = MenuLinkContent::create([
      'title' => 'Parent Link',
      'link' => ['uri' => 'internal:/node/1'],
      'menu_name' => $menu_name,
      'expanded' => TRUE,
    ]);
    $parent->save();

    // 3. Create a child menu link.
    $child = MenuLinkContent::create([
      'title' => 'Child Link',
      'link' => ['uri' => 'internal:/node/2'],
      'menu_name' => $menu_name,
      'parent' => $parent->getPluginId(),
    ]);
    $child->save();

    // 4. Run the helper method.
    $links = $this->menuLinkHelper->getMenuLinksWithTitles($menu_name);

    // 5. Assertions.
    $this->assertCount(2, $links, 'Should return exactly 2 links.');

    // Check Parent Link.
    // Your code strips 'menu_link_content:' prefix.
    $expected_parent_key = $parent->uuid();
    $this->assertEquals($expected_parent_key, $links[0]['key']);
    $this->assertEquals('Parent Link', $links[0]['title']);

    // Check Child Link.
    $expected_child_key = $child->uuid();
    $this->assertEquals($expected_child_key, $links[1]['key']);
    $this->assertEquals('Child Link', $links[1]['title']);
  }

  /**
   * Tests max depth parameter.
   */
  public function testMaxDepth(): void {
    $menu_name = 'depth-test';
    Menu::create(['id' => $menu_name, 'label' => 'Depth Test'])->save();

    $parent = MenuLinkContent::create([
      'title' => 'Level 1',
      'link' => ['uri' => 'internal:/'],
      'menu_name' => $menu_name,
    ]);
    $parent->save();

    MenuLinkContent::create([
      'title' => 'Level 2',
      'link' => ['uri' => 'internal:/'],
      'menu_name' => $menu_name,
      'parent' => $parent->getPluginId(),
    ])->save();

    // Request with max_depth = 1.
    $links = $this->menuLinkHelper->getMenuLinksWithTitles($menu_name, 1);

    $this->assertCount(1, $links, 'Only the top level link should be returned.');
    $this->assertEquals('Level 1', $links[0]['title']);
  }

}
