<?php

namespace Drupal\advanced_menu_condition\Commands;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\StorageInterface;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for Advanced Menu Condition cleanup.
 */
class AdvancedMenuConditionCommands extends DrushCommands {

  /**
   * Constructs a new AdvancedMenuConditionCommands object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Config\StorageInterface $configStorage
   *   The config storage service.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected StorageInterface $configStorage,
  ) {
  }

  /**
   * Clean up advanced_menu_condition from all configurations.
   *
   * @command advanced-menu-condition:cleanup
   * @aliases amc-cleanup
   * @usage drush amc-cleanup
   *   Removes advanced_menu_condition dependencies and visibility
   *   settings from all configurations.
   */
  public function cleanup() {
    if ($this->io()->confirm('This will remove all references to advanced_menu_condition from your configuration. Do you want to continue?')) {
      $total_count = 0;

      // Get all configuration names that might reference the module.
      $all_config_names = $this->configStorage->listAll();

      foreach ($all_config_names as $name) {
        $config = $this->configFactory->getEditable($name);
        $changed = FALSE;

        // Remove module dependency.
        $dependencies = $config->get('dependencies')['module'] ?? [];
        if (in_array('advanced_menu_condition', $dependencies, TRUE)) {
          $dependencies = array_values(array_diff($dependencies, ['advanced_menu_condition']));
          $config->set('dependencies.module', $dependencies);
          $changed = TRUE;
          $this->logger()->notice("Removed module dependency from {$name}");
        }

        // Remove from visibility conditions (common in blocks)
        $visibility = $config->get('visibility') ?? [];
        if (isset($visibility['advanced_menu_position'])) {
          unset($visibility['advanced_menu_position']);
          $config->set('visibility', $visibility);
          $changed = TRUE;
          $this->logger()->notice("Removed visibility condition 'advanced_menu_position' from {$name}");
        }

        if (isset($visibility['advanced_menu_position_not'])) {
          unset($visibility['advanced_menu_position_not']);
          $config->set('visibility', $visibility);
          $changed = TRUE;
          $this->logger()->notice("Removed visibility condition 'advanced_menu_position_not' from {$name}");
        }

        // Remove from conditions.
        $conditions = $config->get('conditions') ?? [];
        if (isset($conditions['advanced_menu_position'])) {
          unset($conditions['advanced_menu_position']);
          $config->set('conditions', $conditions);
          $changed = TRUE;
          $this->logger()->notice("Removed condition 'advanced_menu_position' from {$name}");
        }

        if (isset($conditions['advanced_menu_position_not'])) {
          unset($conditions['advanced_menu_position_not']);
          $config->set('conditions', $conditions);
          $changed = TRUE;
          $this->logger()->notice("Removed condition 'advanced_menu_position_not' from {$name}");
        }

        if ($changed) {
          $config->save();
          $total_count++;
          $this->logger()->success("Cleaned up {$name}");
        }
      }

      $this->logger()->success("Cleaned up {$total_count} configuration(s) total.");
    }
    else {
      $this->io()->writeln('Aborting cleanup.');
      return;
    }
  }

}
