<?php

declare(strict_types=1);

namespace Drupal\advanced_message_subscription\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBundleBase;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\Core\Entity\EntityDeleteForm;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\Plugin\DefaultSingleLazyPluginCollection;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\advanced_message_subscription\AdvancedMessageSubscriptionTypeListBuilder;
use Drupal\advanced_message_subscription\Form\AdvancedMessageSubscriptionTypeForm;

/**
 * Defines the Advanced Message Subscription type configuration entity.
 */
#[ConfigEntityType(
  id: 'adv_message_subscription_type',
  label: new TranslatableMarkup('Advanced Message Subscription type'),
  label_collection: new TranslatableMarkup('Advanced Message Subscription types'),
  label_singular: new TranslatableMarkup('advanced message subscription type'),
  label_plural: new TranslatableMarkup('advanced message subscriptions types'),
  config_prefix: 'adv_message_subscription_type',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
    'uuid' => 'uuid',
  ],
  handlers: [
    'list_builder' => AdvancedMessageSubscriptionTypeListBuilder::class,
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
    'form' => [
      'add' => AdvancedMessageSubscriptionTypeForm::class,
      'edit' => AdvancedMessageSubscriptionTypeForm::class,
      'delete' => EntityDeleteForm::class,
    ],
    'storage' => AdvancedMessageSubscriptionTypeStorage::class,
  ],
  links: [
    'add-form' => '/admin/structure/adv_message_subscription_types/add',
    'edit-form' => '/admin/structure/adv_message_subscription_types/manage/{adv_message_subscription_type}',
    'delete-form' => '/admin/structure/adv_message_subscription_types/manage/{adv_message_subscription_type}/delete',
    'collection' => '/admin/structure/adv_message_subscription_types',
  ],
  admin_permission: 'administer advanced_message_subscription types',
  bundle_of: 'advanced_message_subscription',
  label_count: [
    'singular' => '@count advanced message subscription type',
    'plural' => '@count advanced message subscriptions types',
  ],
  config_export: [
    'id',
    'label',
    'uuid',
    'name_pattern',
    'plugin',
    'configuration',
  ],
)]
class AdvancedMessageSubscriptionType extends ConfigEntityBundleBase {

  /**
   * The plugin collection that holds the block plugin for this entity.
   *
   * @var \Drupal\Core\Plugin\DefaultSingleLazyPluginCollection|null
   */
  protected ?DefaultSingleLazyPluginCollection $pluginCollection = NULL;

  /**
   * The machine name of this advanced message subscription type.
   */
  protected string $id;

  /**
   * The human-readable name of the advanced message subscription type.
   */
  protected string $label;

  /**
   * A pattern used to provide a default name for subscriptions.
   */
  protected string $name_pattern;

  /**
   * The subscription plugin ID.
   */
  protected ?string $plugin = NULL;

  /**
   * The subscription plugin configuration.
   */
  protected array $configuration = [];

  /**
   * Gets the name pattern for this subscription type.
   *
   * @return string|null
   *   The name pattern for this subscription type.
   */
  public function getNamePattern(): ?string {
    return $this->name_pattern ?? NULL;
  }

  /**
   * Returns the plugin ID.
   *
   * @return string|null
   *   The plugin ID for this subscription type.
   */
  public function getPluginId() {
    return $this->plugin;
  }

  /**
   * Returns the plugin instance.
   *
   * @return \Drupal\advanced_message_subscription\AdvancedMessageSubscriptionPluginInterface|null
   *   The plugin instance for this subscription type.
   */
  public function getPlugin() {
    if ($this->plugin) {
      return $this->getPluginCollection()->get($this->plugin);
    }
    return NULL;
  }

  /**
   * Encapsulates the creation of the block's LazyPluginCollection.
   *
   * @return \Drupal\Component\Plugin\LazyPluginCollection
   *   The plugin collection.
   */
  protected function getPluginCollection() {
    if (!$this->pluginCollection) {
      $this->pluginCollection = new DefaultSingleLazyPluginCollection(\Drupal::service('plugin.manager.advanced_message_subscription'), $this->plugin, $this->get('configuration'));
    }
    return $this->pluginCollection;
  }

}
