<?php

declare(strict_types=1);

namespace Drupal\advanced_message_subscription\Entity;

use Drupal\advanced_message_subscription\AdvancedMessageSubscriptionAccessControlHandler;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\Core\Entity\EditorialContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Form\DeleteMultipleForm;
use Drupal\Core\Entity\Form\RevisionDeleteForm;
use Drupal\Core\Entity\Form\RevisionRevertForm;
use Drupal\Core\Entity\Routing\RevisionHtmlRouteProvider;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\advanced_message_subscription\AdvancedMessageSubscriptionInterface;
use Drupal\advanced_message_subscription\AdvancedMessageSubscriptionListBuilder;
use Drupal\advanced_message_subscription\Form\AdvancedMessageSubscriptionForm;
use Drupal\advanced_message_subscription\Routing\AdvancedMessageSubscriptionHtmlRouteProvider;
use Drupal\user\EntityOwnerTrait;
use Drupal\views\EntityViewsData;

/**
 * Defines the advanced message subscription entity class.
 */
#[ContentEntityType(
  id: 'advanced_message_subscription',
  label: new TranslatableMarkup('Advanced Message Subscription'),
  label_collection: new TranslatableMarkup('Advanced Message Subscriptions'),
  label_singular: new TranslatableMarkup('advanced message subscription'),
  label_plural: new TranslatableMarkup('advanced message subscriptions'),
  entity_keys: [
    'id' => 'id',
    'revision' => 'revision_id',
    'bundle' => 'bundle',
    'label' => 'name',
    'owner' => 'uid',
    'published' => 'status',
    'uuid' => 'uuid',
  ],
  handlers: [
    'list_builder' => AdvancedMessageSubscriptionListBuilder::class,
    'access' => AdvancedMessageSubscriptionAccessControlHandler::class,
    'views_data' => EntityViewsData::class,
    'form' => [
      'add' => AdvancedMessageSubscriptionForm::class,
      'edit' => AdvancedMessageSubscriptionForm::class,
      'delete' => ContentEntityDeleteForm::class,
      'delete-multiple-confirm' => DeleteMultipleForm::class,
      'revision-delete' => RevisionDeleteForm::class,
      'revision-revert' => RevisionRevertForm::class,
    ],
    'route_provider' => [
      'html' => AdvancedMessageSubscriptionHtmlRouteProvider::class,
      'revision' => RevisionHtmlRouteProvider::class,
    ],
  ],
  links: [
    'collection' => '/admin/content/advanced-message-subscription',
    'add-form' => '/subscription/add/{adv_message_subscription_type}',
    'add-page' => '/subscription/add',
    'canonical' => '/subscription/{advanced_message_subscription}',
    'edit-form' => '/subscription/{advanced_message_subscription}',
    'delete-form' => '/subscription/{advanced_message_subscription}/delete',
    'delete-multiple-form' => '/admin/content/advanced-message-subscription/delete-multiple',
    'revision' => '/subscription/{advanced_message_subscription}/revision/{adv_message_subscription_rev}/view',
    'revision-delete-form' => '/subscription/{advanced_message_subscription}/revision/{adv_message_subscription_rev}/delete',
    'revision-revert-form' => '/subscription/{advanced_message_subscription}/revision/{adv_message_subscription_rev}/revert',
    'version-history' => '/subscription/{advanced_message_subscription}/revisions',
  ],
  admin_permission: 'administer advanced_message_subscription types',
  bundle_entity_type: 'adv_message_subscription_type',
  bundle_label: new TranslatableMarkup('Advanced Message Subscription type'),
  base_table: 'advanced_message_subscription',
  revision_table: 'adv_message_subscription_rev',
  show_revision_ui: TRUE,
  label_count: [
    'singular' => '@count advanced message subscriptions',
    'plural' => '@count advanced message subscriptions',
  ],
  field_ui_base_route: 'entity.adv_message_subscription_type.edit_form',
  revision_metadata_keys: [
    'revision_user' => 'revision_uid',
    'revision_created' => 'revision_timestamp',
    'revision_log_message' => 'revision_log',
  ],
)]
class AdvancedMessageSubscription extends EditorialContentEntityBase implements AdvancedMessageSubscriptionInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function getType(): AdvancedMessageSubscriptionType {
    return $this->get('bundle')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTagsToInvalidate() {
    return Cache::mergeTags(parent::getCacheTagsToInvalidate(), ["advanced_message_subscription:{$this->bundle()}:{$this->getOwnerId()}"]);
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);
    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the current user the owner.
      $this->setOwnerId(\Drupal::currentUser()->id());
    }
    $this->postCreate($storage);
  }

  /**
   * {@inheritdoc}
   */
  public function postCreate(EntityStorageInterface $storage) {
    $this->getType()->getPlugin()->applySubscriptionData($this);
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Title'))
      ->setRequired(TRUE)
      ->setTranslatable(TRUE)
      ->setRevisionable(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setRevisionable(TRUE)
      ->setLabel(t('Status'))
      ->setDefaultValue(TRUE)
      ->setSetting('on_label', 'Enabled')
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => FALSE,
        ],
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'boolean',
        'label' => 'above',
        'weight' => 0,
        'settings' => [
          'format' => 'enabled-disabled',
        ],
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setRevisionable(TRUE)
      ->setLabel(t('Author'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(self::class . '::getDefaultEntityOwner')
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 15,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setDescription(t('The time that the advanced message subscription was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the advanced message subscription was last edited.'));

    return $fields;
  }

}
