<?php

namespace Drupal\advanced_message_subscription;

use Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscription;
use Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionType;
use Drupal\Component\Plugin\ConfigurableInterface;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Interface for Advanced Message Subscription plugins.
 */
interface AdvancedMessageSubscriptionPluginInterface extends ConfigurableInterface {

  /**
   * Obtains the data required to create the subscription.
   *
   * @param \Drupal\advanced_message_subscription\DataProvider|null $dataProvider
   *   The data provider. Optional, picks up the data provider for the current
   *   request if not provided.
   *
   * @return array
   *   The data.
   */
  public function getRequiredData(?DataProvider $dataProvider = NULL): array;

  /**
   * Checks if the required data for the plugin is available.
   *
   * @param \Drupal\advanced_message_subscription\DataProvider|null $dataProvider
   *   The data provider. Optional, picks up the data provider for the current
   *   request if not provided.
   *
   * @return bool
   *   Indicates if the required data is available.
   */
  public function requiredDataAvailable(?DataProvider $dataProvider = NULL): bool;

  /**
   * Determines if the bundle is set up properly for the plugin.
   *
   * @param \Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionType $subscription_type
   *   The subscription type (bundle).
   *
   * @return bool
   *   Indicates if the bundle is set up properly.
   */
  public function checkBundle(AdvancedMessageSubscriptionType $subscription_type): bool;

  /**
   * Access check for creating a subscription using this plugin.
   *
   * @param \Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionType $subscription_type
   *   The subscription type (bundle).
   * @param \Drupal\advanced_message_subscription\DataProvider|null $dataProvider
   *   The data provider. Optional, picks up the data provider for the current
   *   request if not provided.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function createAccess(AdvancedMessageSubscriptionType $subscription_type, ?DataProvider $dataProvider = NULL): AccessResultInterface;

  /**
   * Set the plugin's data onto the subscription entity.
   *
   * @param \Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscription $subscription
   *   The subscription.
   *
   * @return bool
   *   Indicates if the subscription data was set as expected.
   */
  public function applySubscriptionData(AdvancedMessageSubscription $subscription): bool;

  /**
   * Gets a user's subscription for the given type.
   *
   * @param \Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionType $subscription_type
   *   The subscription type.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   The user account. Optional, defaults to the current user.
   * @param array $data
   *   Data needed for the plugin.
   * @param bool $active_only
   *   Only return an active subscription. Optional, defaults to TRUE.
   *
   * @return \Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscription|null
   *   The subscription, or NULL if it does not exist.
   */
  public function getUserSubscription(AdvancedMessageSubscriptionType $subscription_type, ?AccountInterface $account = NULL, array $data = [], bool $active_only = TRUE): ?AdvancedMessageSubscription;

  /**
   * Returns the cache tags that should be used to invalidate caches.
   *
   * @param \Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscription $subscription
   *   The subscription entity.
   *
   * @return string[]
   *   Set of cache tags.
   *
   * @see \Drupal\Core\Cache\RefinableCacheableDependencyInterface::addCacheTags()
   * @see \Drupal\Core\Cache\CacheableDependencyInterface::getCacheTags()
   */
  public function getCacheTags(AdvancedMessageSubscription $subscription): array;

}
