<?php

namespace Drupal\advanced_message_subscription\Hook;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\advanced_message_subscription\AdvancedMessageSubscriptionPluginManager;
use Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionType;
use Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionTypeStorage;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

/**
 * Hooks for the "entity" subscription plugin.
 */
class EntitySubscriptionHooks {

  use StringTranslationTrait;

  /**
   * Constructs an EntitySubscriptionHooks.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity bundle manager service.
   * @param \Drupal\advanced_message_subscription\AdvancedMessageSubscriptionPluginManager $subscriptionPluginManager
   *   The subscription plugin manager.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Routing\RedirectDestinationInterface $redirectDestination
   *   The redirect destination service.
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    protected readonly AdvancedMessageSubscriptionPluginManager $subscriptionPluginManager,
    protected readonly AccountInterface $currentUser,
    protected readonly RedirectDestinationInterface $redirectDestination,
  ) {
    // No op.
  }

  /**
   * Define extra field elements for subscribing.
   *
   * @return array
   *   Extra field info for displaying the subscription link.
   */
  #[Hook('entity_extra_field_info')]
  public function subscriptionLinkInfo(): array {
    $info = [];

    $storage = $this->entityTypeManager->getStorage('adv_message_subscription_type');
    assert($storage instanceof AdvancedMessageSubscriptionTypeStorage);

    $subscription_types = array_filter($storage->loadByPluginId('entity'), function (AdvancedMessageSubscriptionType $subscription_type) {
      return !empty($subscription_type->getPlugin()->getConfiguration()['extra_field_display']);
    });

    foreach ($subscription_types as $subscription_type) {
      $bundles = $subscription_type->getPlugin()->getConfiguration()['bundles'] ?? [];
      if (empty($bundles)) {
        $bundles = array_keys($this->entityTypeBundleInfo->getBundleInfo($subscription_type->getPlugin()->getConfiguration()['entity_type']));
      }

      foreach ($bundles as $bundle) {
        $info[$subscription_type->getPlugin()->getConfiguration()['entity_type']][$bundle]['display']['advanced_message_subscription_entity__' . $subscription_type->id()] = [
          'label' => $this->t('Entity subscription link'),
          'description' => $this->t('Link to subscribe to or manage entity subscription.'),
          'weight' => 0,
          'visible' => FALSE,
        ];
      }
    }

    return $info;
  }

  /**
   * Displays the entity subscription link.
   *
   * @param array $build
   *   The entity view build array.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param \Drupal\Core\Entity\Display\EntityViewDisplayInterface $display
   *   Teh entity view display.
   * @param string $view_mode
   *   The view mode.
   */
  #[Hook('entity_view')]
  public function displaySubscriptionLinks(array &$build, EntityInterface $entity, EntityViewDisplayInterface $display, string $view_mode): void {
    $storage = $this->entityTypeManager->getStorage('adv_message_subscription_type');
    assert($storage instanceof AdvancedMessageSubscriptionTypeStorage);

    $subscription_types = array_filter($storage->loadByPluginId('entity'), function (AdvancedMessageSubscriptionType $subscription_type) {
      return !empty($subscription_type->getPlugin()->getConfiguration()['extra_field_display']);
    });

    foreach ($subscription_types as $subscription_type) {
      assert($subscription_type instanceof AdvancedMessageSubscriptionType);

      $key = 'advanced_message_subscription_entity__' . $subscription_type->id();
      if (!$display->getComponent($key)) {
        continue;
      }

      $subscription = $subscription_type->getPlugin()->getUserSubscription($subscription_type, $this->currentUser, ['entity' => $entity], FALSE);
      if ($subscription) {
        $url = $subscription->toUrl();
      }
      else {
        $url = Url::fromRoute('entity.advanced_message_subscription.add_form', [
          'adv_message_subscription_type' => $subscription_type->id(),
          'data' => $entity->getEntityTypeId() . ':' . $entity->id(),
        ]);
      }
      $url->setOption('query', $this->redirectDestination->getAsArray());
      $access = $url->access($this->currentUser, TRUE);

      // @todo refactor link generation to a service to avoid code duplication.
      $build[$key] = [
        '#type' => 'link',
        '#title' => $subscription ? $this->t('Manage subscription') : $this->t('Subscribe'),
        '#url' => $url,
        '#access' => $access->isAllowed(),
        '#existing_subscription' => (bool) $subscription,
        '#cache' => [
          'contexts' => ['user'],
          'tags' => [
            "advanced_message_subscription_list:{$subscription_type->id()}:user-{$this->currentUser->id()}",
          ],
        ],
      ];
      $cache = CacheableMetadata::createFromRenderArray($build[$key])
        ->addCacheableDependency($access);
      if ($subscription) {
        $cache->addCacheableDependency($subscription);
      }
      $cache->applyTo($build[$key]);
    }
  }

}
