<?php

namespace Drupal\advanced_message_subscription;

use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionType;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;

/**
 * Generates links for message subscriptions.
 */
class LinkGenerator {

  /**
   * Constructs a LinkGenerator object.
   *
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Routing\RedirectDestinationInterface $redirectDestination
   *   The redirect destination service.
   */
  public function __construct(
    protected readonly AccountInterface $currentUser,
    protected readonly RedirectDestinationInterface $redirectDestination,
  ) {
    // No op.
  }

  /**
   * Gets a render array for a link to add or manage a subscription.
   *
   * @param \Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionType $subscription_type
   *   The subscription type.
   * @param array $query_data
   *   The query data for obtianing an existing subscription.
   * @param string|null $link_data
   *   The link data for creating the subscription, if any.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   The user account to get the subscription for, defaults to the current
   *   user.
   *
   * @return array
   *   The render array for the link.
   */
  public function getLinkArray(AdvancedMessageSubscriptionType $subscription_type, array $query_data, ?string $link_data = NULL, ?AccountInterface $account = NULL): array {
    if (!$account) {
      $account = $this->currentUser;
    }

    $subscription = $subscription_type->getPlugin()->getUserSubscription($subscription_type, $account, $query_data, FALSE);
    if ($subscription) {
      $url = $subscription->toUrl();
    }
    else {
      $url = Url::fromRoute('entity.advanced_message_subscription.add_form', ['adv_message_subscription_type' => $subscription_type->id()]);
      if ($link_data) {
        $url->setRouteParameter('data', $link_data);
      }
    }
    $url->setOption('query', $this->redirectDestination->getAsArray());
    $access = $url->access($this->currentUser, TRUE);

    $link = [
      '#type' => 'link',
      '#title' => $subscription ? $subscription_type->getManageLinkText() : $subscription_type->getAddLinkText(),
      '#url' => $url,
      '#access' => $access->isAllowed(),
      '#existing_subscription' => (bool) $subscription,
      '#cache' => [
        'contexts' => ['user'],
        'tags' => [
          "advanced_message_subscription_list:{$subscription_type->id()}:user-{$this->currentUser->id()}",
        ],
      ],
    ];

    $cache = CacheableMetadata::createFromRenderArray($link)
      ->addCacheableDependency($access);
    if ($subscription) {
      $cache->addCacheableDependency($subscription);
    }
    $cache->applyTo($link);
    return $link;
  }

}
