<?php

declare(strict_types=1);

namespace Drupal\advanced_message_subscription;

use Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscription;
use Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionType;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Base class for advanced_message_subscription plugins.
 */
abstract class AdvancedMessageSubscriptionPluginBase extends PluginBase implements AdvancedMessageSubscriptionPluginInterface, PluginFormInterface {

  use DependencySerializationTrait;
  use MessengerTrait;
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->setConfiguration($configuration);
  }

  /**
   * {@inheritdoc}
   */
  public function label(): string {
    // Cast the label to a string since it is a TranslatableMarkup object.
    return (string) $this->pluginDefinition['label'];
  }

  /**
   * Gets the specified field from a bundle.
   *
   * @param string $bundle
   *   The bundle.
   * @param string $field_name
   *   The field name.
   *
   * @return \Drupal\Core\Field\FieldDefinitionInterface|null
   *   The field, or NULL if not available.
   */
  protected function getBundleField(string $bundle, string $field_name): ?FieldDefinitionInterface {
    $fields = $this->entityFieldManager()
      ->getFieldDefinitions('advanced_message_subscription', $bundle);
    $field = $fields[$field_name] ?? NULL;
    return ($field instanceof FieldDefinitionInterface) ? $field : NULL;
  }

  /**
   * Get the current user.
   *
   * @return \Drupal\Core\Session\AccountProxyInterface
   *   The current user.
   */
  protected function currentUser(): AccountProxyInterface {
    return \Drupal::currentUser();
  }

  /**
   * Get the data provider service.
   *
   * @return \Drupal\advanced_message_subscription\DataProvider
   *   The data provider service.
   */
  protected function dataProvider(): DataProvider {
    return \Drupal::service('advanced_message_subscription.data_provider');
  }

  /**
   * Get the entity field manager service.
   *
   * @return \Drupal\Core\Entity\EntityFieldManagerInterface
   *   The entity field manager service.
   */
  protected function entityFieldManager(): EntityFieldManagerInterface {
    return \Drupal::service('entity_field.manager');
  }

  /**
   * Get the entity type bundle info service.
   *
   * @return \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   *   The entity type bundle info service.
   */
  protected function entityTypeBundleInfo(): EntityTypeBundleInfoInterface {
    return \Drupal::service('entity_type.bundle.info');
  }

  /**
   * Get the module handler service.
   *
   * @return \Drupal\Core\Extension\ModuleHandlerInterface
   *   The module handler service.
   */
  protected function moduleHandler(): ModuleHandlerInterface {
    return \Drupal::moduleHandler();
  }

  /**
   * Get the entity type manager service.
   *
   * @return \Drupal\Core\Entity\EntityTypeManagerInterface
   *   The entity type manager service.
   */
  protected function entityTypeManager(): EntityTypeManagerInterface {
    return \Drupal::entityTypeManager();
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration() {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration) {
    $this->configuration = NestedArray::mergeDeep($this->defaultConfiguration(), $configuration);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function createAccess(AdvancedMessageSubscriptionType $subscription_type, ?DataProvider $dataProvider = NULL): AccessResultInterface {
    return AccessResult::allowedIf($this->checkBundle($subscription_type) && $this->requiredDataAvailable($dataProvider))
      ->addCacheableDependency($subscription_type);
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags(AdvancedMessageSubscription $subscription): array {
    return [];
  }

  /**
   * Build a form element to configure handling events.
   *
   * @return array
   *   The form element.
   */
  protected function buildEventTypesFormElement(): array {
    $element = [
      '#type' => 'fieldset',
      '#title' => $this->t('Event types'),
      '#description' => $this->t('Select event types to handle and messaging for each.'),
    ];

    $message_templates = $this->entityTypeManager()->getStorage('message_template')->loadMultiple();

    foreach ($this->getEventTypes() as $event_type => $label) {
      $element[$event_type]['status'] = [
        '#type' => 'checkbox',
        '#title' => $label,
        '#default_value' => $this->getConfiguration()['event_types'][$event_type]['status'] ?? FALSE,
      ];
      $selector = [':input[name="configuration[event_types][' . $event_type . '][status]"]' => ['checked' => TRUE]];
      $element[$event_type]['message_template'] = [
        '#type' => 'select',
        '#title' => $this->t('Message template'),
        '#default_value' => $this->getConfiguration()['event_types'][$event_type]['message_template'] ?? FALSE,
        '#options' => array_map(fn ($msg) => $msg->label(), $message_templates),
        '#states' => [
          'visible' => $selector,
          'required' => $selector,
        ],
      ];
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function getEventMessageTemplate(AdvancedMessageSubscriptionEvent $event): ?string {
    if (empty($this->configuration['event_types'][$event->getEventType()]['status'])) {
      return NULL;
    }
    return $this->configuration['event_types'][$event->getEventType()]['message_template'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['event_types'] = $this->buildEventTypesFormElement();
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    // No op.
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    // No op.
  }

}
