<?php

declare(strict_types=1);

namespace Drupal\advanced_message_subscription\Entity;

use Drupal\advanced_message_subscription\AdvancedMessageSubscriptionTypeAccessControlHandler;
use Drupal\advanced_message_subscription\DataProvider;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Config\Entity\ConfigEntityBundleBase;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\Core\Entity\EntityDeleteForm;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\Plugin\DefaultSingleLazyPluginCollection;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\advanced_message_subscription\AdvancedMessageSubscriptionTypeListBuilder;
use Drupal\advanced_message_subscription\Form\AdvancedMessageSubscriptionTypeForm;

/**
 * Defines the Advanced Message Subscription type configuration entity.
 */
#[ConfigEntityType(
  id: 'adv_message_subscription_type',
  label: new TranslatableMarkup('Subscription type'),
  label_collection: new TranslatableMarkup('Subscription types'),
  label_singular: new TranslatableMarkup('advanced message subscription type'),
  label_plural: new TranslatableMarkup('advanced message subscriptions types'),
  config_prefix: 'adv_message_subscription_type',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
    'uuid' => 'uuid',
  ],
  handlers: [
    'list_builder' => AdvancedMessageSubscriptionTypeListBuilder::class,
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
    'form' => [
      'add' => AdvancedMessageSubscriptionTypeForm::class,
      'edit' => AdvancedMessageSubscriptionTypeForm::class,
      'delete' => EntityDeleteForm::class,
    ],
    'storage' => AdvancedMessageSubscriptionTypeStorage::class,
    'access' => AdvancedMessageSubscriptionTypeAccessControlHandler::class,
  ],
  links: [
    'add-form' => '/admin/structure/adv_message_subscription_types/add',
    'edit-form' => '/admin/structure/adv_message_subscription_types/manage/{adv_message_subscription_type}',
    'delete-form' => '/admin/structure/adv_message_subscription_types/manage/{adv_message_subscription_type}/delete',
    'collection' => '/admin/structure/adv_message_subscription_types',
  ],
  admin_permission: 'administer advanced_message_subscription types',
  bundle_of: 'advanced_message_subscription',
  label_count: [
    'singular' => '@count advanced message subscription type',
    'plural' => '@count advanced message subscriptions types',
  ],
  config_export: [
    'id',
    'label',
    'uuid',
    'name_pattern',
    'add_link_text',
    'manage_link_text',
    'notify',
    'plugin',
    'configuration',
  ],
)]
class AdvancedMessageSubscriptionType extends ConfigEntityBundleBase {

  /**
   * The plugin collection that holds the block plugin for this entity.
   *
   * @var \Drupal\Core\Plugin\DefaultSingleLazyPluginCollection|null
   */
  protected ?DefaultSingleLazyPluginCollection $pluginCollection = NULL;

  /**
   * The machine name of this advanced message subscription type.
   */
  protected string $id;

  /**
   * The human-readable name of the advanced message subscription type.
   */
  protected string $label;

  /**
   * A pattern used to provide a default name for subscriptions.
   */
  protected string $name_pattern;

  /**
   * The subscription plugin ID.
   */
  protected ?string $plugin = NULL;

  /**
   * The subscription plugin configuration.
   */
  protected array $configuration = [];

  /**
   * Gets the name pattern for this subscription type.
   *
   * @return string|null
   *   The name pattern for this subscription type.
   */
  public function getNamePattern(): ?string {
    return $this->name_pattern ?? NULL;
  }

  /**
   * Gets the text for an add link for this subscription type.
   *
   * @return string|null
   *   The add link text.
   */
  public function getAddLinkText(): ?string {
    return $this->add_link_text ?? NULL;
  }

  /**
   * Gets the text for a manage link for this subscription type.
   *
   * @return string|null
   *   The manage link text.
   */
  public function getManageLinkText(): ?string {
    return $this->manage_link_text ?? NULL;
  }

  /**
   * Gets the notify configuration.
   *
   * @return bool
   *   The notify configuration.
   */
  public function notify(): bool {
    return (bool) $this->get('notify') ?? FALSE;
  }

  /**
   * Returns the plugin ID.
   *
   * @return string|null
   *   The plugin ID for this subscription type.
   */
  public function getPluginId() {
    return $this->plugin;
  }

  /**
   * Set the plugin ID.
   *
   * @param string $plugin_id
   *   The plugin ID.
   *
   * @return $this
   */
  public function setPluginId($plugin_id) {
    $this->plugin_id = $plugin_id;
    return $this;
  }

  /**
   * Returns the plugin instance.
   *
   * @return \Drupal\advanced_message_subscription\AdvancedMessageSubscriptionPluginInterface|null
   *   The plugin instance for this subscription type.
   */
  public function getPlugin() {
    if ($this->plugin) {
      return $this->getPluginCollection()->get($this->plugin);
    }
    return NULL;
  }

  /**
   * Encapsulates the creation of the block's LazyPluginCollection.
   *
   * @return \Drupal\Component\Plugin\LazyPluginCollection
   *   The plugin collection.
   */
  protected function getPluginCollection() {
    if (!$this->pluginCollection) {
      $this->pluginCollection = new DefaultSingleLazyPluginCollection(\Drupal::service('plugin.manager.advanced_message_subscription'), $this->plugin, $this->get('configuration'));
    }
    return $this->pluginCollection;
  }

  /**
   * Checks access for creating a new subscription of this type.
   *
   * @param \Drupal\advanced_message_subscription\DataProvider|null $dataProvider
   *   (optional) A data provider to check required data availability.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function createAccess(?DataProvider $dataProvider = NULL): AccessResultInterface {
    $plugin = $this->getPlugin();
    if (!$plugin) {
      return AccessResult::forbidden();
    }
    return $plugin->createAccess($this, $dataProvider)
      ->addCacheableDependency($this)
      ->addCacheContexts(['user']);
  }

}
