<?php

declare(strict_types=1);

namespace Drupal\advanced_message_subscription\Form;

use Drupal\advanced_message_subscription\AdvancedMessageSubscriptionPluginManager;
use Drupal\Core\Entity\BundleEntityFormBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionType;
use Drupal\Core\Plugin\PluginFormInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for advanced message subscription type forms.
 */
class AdvancedMessageSubscriptionTypeForm extends BundleEntityFormBase {

  /**
   * The advanced message subscription type entity being edited.
   *
   * @var \Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionType
   */
  protected $entity;

  /**
   * Constructs a AdvancedMessageSubscriptionTypeForm.
   *
   * @param \Drupal\advanced_message_subscription\AdvancedMessageSubscriptionPluginManager $subscriptionPluginManager
   *   The advanced message subscription plugin manager.
   */
  public function __construct(
    protected AdvancedMessageSubscriptionPluginManager $subscriptionPluginManager,
  ) {
    // No op.
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.advanced_message_subscription'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    $subscription_type = $this->entity;
    assert($subscription_type instanceof AdvancedMessageSubscriptionType);

    if ($this->operation === 'edit') {
      $form['#title'] = $this->t('Edit %label advanced message subscription type', ['%label' => $subscription_type->label()]);
    }

    $form['label'] = [
      '#title' => $this->t('Label'),
      '#type' => 'textfield',
      '#default_value' => $subscription_type->label(),
      '#description' => $this->t('The human-readable name of this advanced message subscription type.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $subscription_type->id(),
      '#maxlength' => EntityTypeInterface::BUNDLE_MAX_LENGTH,
      '#machine_name' => [
        'exists' => [AdvancedMessageSubscriptionType::class, 'load'],
        'source' => ['label'],
      ],
      '#description' => $this->t('A unique machine-readable name for this advanced message subscription type. It must only contain lowercase letters, numbers, and underscores.'),
    ];

    $form['name_pattern'] = [
      '#title' => $this->t('Name pattern'),
      '#type' => 'textfield',
      '#default_value' => $subscription_type->getNamePattern(),
      '#description' => $this->t('A pattern used to provide a default name for subscriptions.'),
      '#required' => TRUE,
      '#element_validate' => ['token_element_validate'],
      '#after_build' => ['token_element_validate'],
      '#token_types' => ['advanced_message_subscription'],
    ];

    // Show the token help relevant to this pattern type.
    $form['token_help'] = [
      '#theme' => 'token_tree_link',
      '#token_types' => ['advanced_message_subscription'],
    ];

    $form['add_link_text'] = [
      '#title' => $this->t('Add link text'),
      '#type' => 'textfield',
      '#default_value' => $subscription_type->getAddLinkText(),
      '#description' => $this->t('The text to use for add subscription links.'),
      '#required' => TRUE,
    ];

    $form['manage_link_text'] = [
      '#title' => $this->t('Manage link text'),
      '#type' => 'textfield',
      '#default_value' => $subscription_type->getManageLinkText(),
      '#description' => $this->t('The text to use for manage subscription links.'),
      '#required' => TRUE,
    ];

    $form['notify'] = [
      '#title' => $this->t('Notify'),
      '#type' => 'checkbox',
      '#default_value' => $subscription_type->get('notify'),
      '#description' => $this->t('Send notifications upon message creation. Requires the message_notify module.'),
    ];

    $options = array_map(fn($definition) => $definition['label'], $this->subscriptionPluginManager->getDefinitions());
    $form['plugin'] = [
      '#type' => 'select',
      '#title' => $this->t('Plugin'),
      '#options' => $options,
      '#default_value' => $subscription_type->getPluginId(),
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::updatePluginConfig',
        'wrapper' => 'edit-plugin-config-wrapper',
      ],
    ];

    if ($plugin_id = $form_state->getValue('plugin')) {
      $subscription_type->setPluginId($plugin_id);
    }
    $form_state->set('initial_plugin', $subscription_type->getPluginId());

    if ($plugin = $subscription_type->getPlugin()) {
      $form['configuration'] = [
        '#tree' => TRUE,
        '#title' => $this->t('Plugin configuration'),
        '#type' => 'fieldset',
        '#attributes' => ['id' => 'edit-plugin-config-wrapper'],
      ];

      if ($plugin instanceof PluginFormInterface) {
        $form['configuration'] = $plugin->buildConfigurationForm($form['configuration'], $form_state);
      }
    }
    else {
      $form['configuration'] = [
        '#type' => 'container',
        '#attributes' => ['id' => 'edit-plugin-config-wrapper'],
      ];
    }

    return $this->protectBundleIdElement($form);
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state): array {
    $actions = parent::actions($form, $form_state);
    $actions['submit']['#value'] = $this->t('Save subscription type');
    $actions['delete']['#value'] = $this->t('Delete subscription type');
    return $actions;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $plugin_id = $form_state->getValue('plugin');
    if ($form_state->get('initial_plugin') != $plugin_id) {
      $form_state->setRebuild();
    }

    $subscription_type = $this->entity;
    $subscription_type->setPluginId($plugin_id);

    assert($subscription_type instanceof AdvancedMessageSubscriptionType);
    if ($plugin = $subscription_type->getPlugin()) {
      if ($plugin instanceof PluginFormInterface) {
        $plugin->validateConfigurationForm($form, $form_state);
      }
      else {
        $form_state->setValue('configuration', []);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);

    $subscription_type = $this->entity;
    assert($subscription_type instanceof AdvancedMessageSubscriptionType);
    if ($plugin = $subscription_type->getPlugin()) {
      if ($plugin instanceof PluginFormInterface) {
        $plugin->submitConfigurationForm($form, $form_state);
      }
    }

    $message_args = ['%label' => $this->entity->label()];
    $this->messenger()->addStatus(
      match($result) {
        SAVED_NEW => $this->t('The advanced message subscription type %label has been added.', $message_args),
        SAVED_UPDATED => $this->t('The advanced message subscription type %label has been updated.', $message_args),
      }
    );
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));

    return $result;
  }

  /**
   * Handles plugin configuration portion of the form via AJAX.
   */
  public function updatePluginConfig(array $form) {
    return $form['configuration'];
  }

}
