<?php

declare(strict_types=1);

namespace Drupal\advanced_message_subscription;

use Drupal\advanced_message_subscription\Message\MessageCreationInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\message\MessageInterface;
use Drupal\message\MessageTemplateInterface;

/**
 * Processes subscriptions on triggering events.
 */
class SubscriptionProcessor {

  /**
   * Subscription type storage.
   *
   * @var \Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionTypeStorage
   */
  protected $subscriptionTypeStorage;

  /**
   * Constructs a SubscriptionProcessor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Queue\QueueFactory $queueFactory
   *   The queue factory service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly QueueFactory $queueFactory,
    protected readonly ModuleHandlerInterface $moduleHandler,
  ) {
    $this->subscriptionTypeStorage = $entityTypeManager->getStorage('adv_message_subscription_type');
  }

  /**
   * Process an event that may produce messages.
   *
   * @param AdvancedMessageSubscriptionEvent $event
   *   The event.
   */
  public function processEvent(AdvancedMessageSubscriptionEvent $event): void {
    $this->moduleHandler->invokeAll('advanced_message_subscription_event_process', [$event]);
    if (!$event->canProcess()) {
      return;
    }

    // Process applicable subscription types, per plugin ID.
    $subscription_types = $this->subscriptionTypeStorage->loadByPluginId($event->getPluginId());

    foreach ($subscription_types as $subscription_type) {
      // Ensure we can get the plugin.
      if (!$plugin = $subscription_type->getPlugin()) {
        continue;
      }

      // Hopefully our plugin already checked that we have a template for use,
      // but we might as well check it here too anyway as we'll need to pass
      // the template on to the queue.
      if (!$message_template = $plugin->getEventMessageTemplate($event)) {
        continue;
      }

      // Obtain our queue.
      $queue = $this->queueFactory->get('advanced_message_subscription_message');

      // Add an item for each subscription that the plugin identifies as
      // applicable to the event.
      foreach ($plugin->findSubscriptions($event, $subscription_type) as $subscription) {
        $queue->createItem([
          'plugin_id' => $event->getPluginId(),
          'event_type' => $event->getEventType(),
          'timestamp' => $event->getTimestamp(),
          'entity_id' => $event->getEntity()?->id(),
          'entity_type_id' => $event->getEntity()?->getEntityTypeId(),
          'entity_label' => $event->getEntity()?->label(),
          'uid' => $event->getAccount()->id(),
          'data' => $event->getData(),
          'message_template' => $message_template,
          'subscription_id' => $subscription,
        ]);
      }
    }
  }

  /**
   * Creates a message for a subscription.
   *
   * @param \Drupal\message\MessageTemplateInterface $message_template
   *   The message template.
   * @param AdvancedMessageSubscriptionInterface $subscription
   *   The subscription.
   * @param array $data
   *   Event data.
   *
   * @return \Drupal\message\MessageInterface|null
   *   The message, or NULL if no message created.
   */
  public function createMessage(MessageTemplateInterface $message_template, AdvancedMessageSubscriptionInterface $subscription, array $data): ?MessageInterface {
    if (!$plugin = $subscription->getType()->getPlugin()) {
      return NULL;
    }

    $message = $this->entityTypeManager
      ->getStorage('message')
      ->create([
        'template' => $message_template->id(),
        'uid' => $subscription->getOwnerId(),
        'created' => $data['timestamp'],
      ]);
    assert($message instanceof MessageInterface);

    if ($plugin instanceof MessageCreationInterface) {
      $plugin->initializeNewMessage($message, $subscription, $data);
    }

    $params = [$message, $subscription, $data];
    $this->moduleHandler->invokeAll('advanced_message_subscription_message_create', $params);

    $message->save();

    // Send notification if subscription type is configured to do so, and we
    // have the message_notify module.
    if ($subscription->getType()->notify() && $this->moduleHandler->moduleExists('message_notify')) {
      // Allow other modules to cancel notification.
      $result = $this->moduleHandler->invokeAll('message_notify_subscription_notify_cancel', $params);
      $result = array_filter($result);
      if (count($result) == 0) {
        \Drupal::service('message_notify.sender')->send($message);
      }
    }

    return $message;
  }

}
