<?php

namespace Drupal\advanced_message_subscription;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Represents an event that may trigger subscriptions.
 */
class AdvancedMessageSubscriptionEvent {

  /**
   * Indicates if the event should be processed.
   *
   * @var bool
   */
  protected bool $process = TRUE;

  /**
   * The timestamp from when the event was initiated.
   *
   * @var int
   */
  protected int $timestamp;

  /**
   * Constructs an AdvancedMessageSubscriptionEvent.
   *
   * @param string $pluginId
   *   The plugin ID of the event.
   * @param string $eventType
   *   The event type, usually an operation like insert/update/delete.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The pertinent entity for the event, but optional as events don't have
   *   to be entity-based.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   The user who initiated the event. Optional, uses the current user if
   *   omitted.
   * @param array $data
   *   Other data applicable to the event.
   */
  public function __construct(
    protected readonly string $pluginId,
    protected readonly string $eventType,
    protected readonly EntityInterface $entity,
    protected ?AccountInterface $account = NULL,
    protected readonly array $data = [],
  ) {
    if (!$this->account) {
      $this->account = \Drupal::currentUser();
    }
    $this->timestamp = \Drupal::time()->getCurrentTime();
  }

  /**
   * Get the plugin ID.
   *
   * @return string
   *   The plugin ID.
   */
  public function getPluginId(): string {
    return $this->pluginId;
  }

  /**
   * Get the event type.
   *
   * @return string
   *   The event type.
   */
  public function getEventType(): string {
    return $this->eventType;
  }

  /**
   * Get the entity.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The entity.
   */
  public function getEntity(): ?EntityInterface {
    return $this->entity;
  }

  /**
   * Gets the account that initiated the event.
   *
   * @return \Drupal\Core\Session\AccountInterface|null
   *   The account that initiated the event.
   */
  public function getAccount(): ?AccountInterface {
    return $this->account;
  }

  /**
   * Get event data.
   *
   * @param string|null $key
   *   The key of the event data to return. Optional, omit to return all event
   *   data.
   *
   * @return mixed
   *   The specified event data, or NULL if not available. If no key is
   *   specified, all event data will be returned as an array.
   */
  public function getData(?string $key = NULL): mixed {
    if ($key === NULL) {
      return $this->data;
    }
    return $this->data[$key] ?? NULL;
  }

  /**
   * Cancel the event from processing.
   *
   * @return $this
   */
  public function cancel(): self {
    $this->process = FALSE;
    return $this;
  }

  /**
   * Determines  if the event can process.
   *
   * @return bool
   *   Indicates if the event can process.
   */
  public function canProcess(): bool {
    return $this->process;
  }

  /**
   * Get the timestamp from when the event was initiated.
   *
   * @return int
   *   The timestamp from when the event was initiated.
   */
  public function getTimestamp(): int {
    return $this->timestamp;
  }

}
