<?php

namespace Drupal\advanced_message_subscription\Hook;

use Drupal\advanced_message_subscription\AdvancedMessageSubscriptionEvent;
use Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionTypeStorage;
use Drupal\advanced_message_subscription\LinkGenerator;
use Drupal\advanced_message_subscription\SubscriptionProcessor;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionType;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Hooks for the "entity" subscription plugin.
 */
class EntitySubscriptionHooks {

  use StringTranslationTrait;

  /**
   * Subscription type storage.
   *
   * @var \Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionTypeStorage
   */
  protected AdvancedMessageSubscriptionTypeStorage $subscriptionTypeStorage;

  /**
   * Constructs an EntitySubscriptionHooks.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity bundle manager service.
   * @param \Drupal\advanced_message_subscription\LinkGenerator $linkGenerator
   *   The link generator service.
   * @param \Drupal\advanced_message_subscription\SubscriptionProcessor $subscriptionProcessor
   *   The subscription processor service.
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    protected readonly LinkGenerator $linkGenerator,
    protected readonly SubscriptionProcessor $subscriptionProcessor,
  ) {
    $this->subscriptionTypeStorage = $this->entityTypeManager->getStorage('adv_message_subscription_type');
  }

  /**
   * Define extra field elements for subscribing.
   *
   * @return array
   *   Extra field info for displaying the subscription link.
   */
  #[Hook('entity_extra_field_info')]
  public function subscriptionLinkInfo(): array {
    $info = [];

    $subscription_types = array_filter($this->subscriptionTypeStorage->loadByPluginId('entity'), function (AdvancedMessageSubscriptionType $subscription_type) {
      return !empty($subscription_type->getPlugin()->getConfiguration()['extra_field_display']);
    });

    foreach ($subscription_types as $subscription_type) {
      $bundles = $subscription_type->getPlugin()->getConfiguration()['bundles'] ?? [];
      if (empty($bundles)) {
        $bundles = array_keys($this->entityTypeBundleInfo->getBundleInfo($subscription_type->getPlugin()->getConfiguration()['entity_type']));
      }

      foreach ($bundles as $bundle) {
        $info[$subscription_type->getPlugin()->getConfiguration()['entity_type']][$bundle]['display']['advanced_message_subscription_entity__' . $subscription_type->id()] = [
          'label' => $this->t('Entity subscription link'),
          'description' => $this->t('Link to subscribe to or manage entity subscription.'),
          'weight' => 0,
          'visible' => FALSE,
        ];
      }
    }

    return $info;
  }

  /**
   * Displays the entity subscription link.
   *
   * @param array $build
   *   The entity view build array.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param \Drupal\Core\Entity\Display\EntityViewDisplayInterface $display
   *   Teh entity view display.
   * @param string $view_mode
   *   The view mode.
   */
  #[Hook('entity_view')]
  public function displaySubscriptionLinks(array &$build, EntityInterface $entity, EntityViewDisplayInterface $display, string $view_mode): void {
    $subscription_types = array_filter($this->subscriptionTypeStorage->loadByPluginId('entity'), function (AdvancedMessageSubscriptionType $subscription_type) {
      return !empty($subscription_type->getPlugin()->getConfiguration()['extra_field_display']);
    });

    foreach ($subscription_types as $subscription_type) {
      assert($subscription_type instanceof AdvancedMessageSubscriptionType);

      $key = 'advanced_message_subscription_entity__' . $subscription_type->id();
      if (!$display->getComponent($key)) {
        continue;
      }

      $query_data = ['entity' => $entity];
      $link_data = $entity->getEntityTypeId() . ':' . $entity->id();
      $build[$key] = $this->linkGenerator->getLinkArray($subscription_type, $query_data, $link_data);
    }
  }

  /**
   * Handle subscriptions for entity insertion.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   */
  #[Hook('entity_insert')]
  public function entityInsert(EntityInterface $entity) {
    $event = new AdvancedMessageSubscriptionEvent('entity', 'insert', $entity);
    $this->subscriptionProcessor->processEvent($event);
  }

  /**
   * Handle subscriptions for entity updates.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   */
  #[Hook('entity_update')]
  public function entityUpdate(EntityInterface $entity) {
    $event = new AdvancedMessageSubscriptionEvent('entity', 'update', $entity);
    $this->subscriptionProcessor->processEvent($event);
  }

  /**
   * Handle subscriptions for entity deletions.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   */
  #[Hook('entity_delete')]
  public function entityDelete(EntityInterface $entity) {
    $event = new AdvancedMessageSubscriptionEvent('entity', 'delete', $entity);
    $this->subscriptionProcessor->processEvent($event);
  }

}
