<?php

declare(strict_types=1);

namespace Drupal\advanced_message_subscription\Plugin\QueueWorker;

use Drupal\advanced_message_subscription\AdvancedMessageSubscriptionInterface;
use Drupal\advanced_message_subscription\SubscriptionProcessor;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\message\MessageTemplateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines 'advanced_message_subscription_message' queue worker.
 */
#[QueueWorker(
  id: 'advanced_message_subscription_message',
  title: new TranslatableMarkup('Subscription Message'),
  cron: ['time' => 60],
)]
class SubscriptionMessageQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Subscription processor.
   *
   * @var \Drupal\advanced_message_subscription\SubscriptionProcessor
   */
  protected SubscriptionProcessor $subscriptionProcessor;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->subscriptionProcessor = $container->get(SubscriptionProcessor::class);
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    $message_template = $this->entityTypeManager->getStorage('message_template')
      ->load($data['message_template']);
    if (!$message_template instanceof MessageTemplateInterface) {
      return;
    }

    $subscription = $this->entityTypeManager->getStorage('advanced_message_subscription')
      ->load($data['subscription_id']);
    if (!$subscription instanceof AdvancedMessageSubscriptionInterface) {
      return;
    }

    $this->subscriptionProcessor->createMessage($message_template, $subscription, $data);
  }

}
