<?php

declare(strict_types=1);

namespace Drupal\advanced_message_subscription\Plugin\AdvancedMessageSubscription;

use Drupal\advanced_message_subscription\DataProvider;
use Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscriptionType;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\ContentEntityType;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\advanced_message_subscription\AdvancedMessageSubscriptionPluginBase;
use Drupal\advanced_message_subscription\Attribute\AdvancedMessageSubscription;
use Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscription as AdvancedMessageSubscriptionEntity;

/**
 * Plugin implementation of the advanced_message_subscription.
 */
#[AdvancedMessageSubscription(
  id: 'entity',
  label: new TranslatableMarkup('Entity'),
  description: new TranslatableMarkup('Subscribe to an entity.'),
)]
class Entity extends AdvancedMessageSubscriptionPluginBase implements PluginFormInterface {

  /**
   * {@inheritdoc}
   */
  public function getRequiredData(?DataProvider $dataProvider = NULL): array {
    if (!$dataProvider) {
      $dataProvider = $this->dataProvider();
    }

    $data = ['entity' => NULL];
    $parts = explode(':', $dataProvider->getDataParam() ?? '', 2);
    $entity_type = array_shift($parts);
    $entity_id = array_shift($parts);

    if (!$entity_type || !$entity_id) {
      return $data;
    }

    $configured_entity_type = $this->configuration['entity_type'] ?? NULL;
    if ($entity_type !== $configured_entity_type) {
      return $data;
    }

    $entity = $this->entityTypeManager()->getStorage($entity_type)
      ->load($entity_id);
    if (!$entity) {
      return $data;
    }

    $bundles = $this->configuration['bundles'] ?? [];
    if ($bundles && !in_array($entity->bundle(), $bundles)) {
      return $data;
    }

    $data['entity'] = $entity;
    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function requiredDataAvailable(?DataProvider $dataProvider = NULL): bool {
    $data = $this->getRequiredData($dataProvider);
    return !empty($data['entity']) && ($data['entity'] instanceof ContentEntityInterface);
  }

  /**
   * {@inheritdoc}
   */
  public function checkBundle(AdvancedMessageSubscriptionType $subscription_type): bool {
    if (!$field_name = $this->configuration['field_name'] ?? NULL) {
      return FALSE;
    }

    if (!$field = $this->getBundleField($subscription_type->id(), $field_name)) {
      return FALSE;
    }

    return $field->getType() == 'entity_reference';
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['field_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Field name'),
      '#default_value' => $this->configuration['field_name'] ?? NULL,
      '#description' => $this->t('The subscription type will be required to have an entity reference field referencing the specified entity type. Enter the machine name of the field.'),
      '#required' => TRUE,
    ];

    $options = array_map(function ($definition) {
      if (!$definition instanceof ContentEntityType) {
        return FALSE;
      }
      return $definition->getLabel();
    }, $this->entityTypeManager()->getDefinitions());
    $options = array_filter($options);

    $entity_type = $form_state->getValue(['configuration', 'entity_type']) ?: $this->configuration['entity_type'] ?? NULL;
    $form['entity_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity Type'),
      '#options' => $options,
      '#default_value' => $entity_type,
      '#description' => $this->t('Specify the entity type for the subscription.'),
      '#required' => TRUE,
      '#ajax' => [
        'callback' => [$this, 'updateBundlesAjax'],
        'wrapper' => 'edit-configuration-bundles-wrapper',
      ],
    ];

    if ($entity_type) {
      $options = array_map(function ($bundle) {
        return $bundle['label'];
      }, $this->entityTypeBundleInfo()->getBundleInfo($entity_type));

      $form['bundles'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Bundles'),
        '#options' => $options,
        '#default_value' => $this->configuration['bundles'] ?? [],
        '#description' => $this->t('Restrict to specific bundles. If none are selected, all bundles are allowed.'),
        '#access' => !empty($options),
        '#prefix' => '<div id="edit-configuration-bundles-wrapper">',
        '#suffix' => '</div>',
      ];
    }
    else {
      $form['bundles'] = [
        '#type' => 'container',
        '#attributes' => ['id' => 'edit-configuration-bundles-wrapper'],
      ];
    }

    $form['extra_field_display'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Extra field display'),
      '#description' => $this->t('Provides an "extra field" which can be displayed using the entity\'s "Manage Display" configuration.'),
      '#default_value' => $this->configuration['extra_field_display'] ?? NULL,
    ];

    $form['access_check'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Check access'),
      '#default_value' => $this->configuration['access_check'] ?? NULL,
    ];

    $form['access_operation'] = [
      '#type' => 'select',
      '#title' => $this->t('Access check operation'),
      '#options' => [
        'view' => $this->t('View'),
        'edit' => $this->t('Edit'),
        'delete' => $this->t('Delete'),
      ],
      '#default_value' => $this->configuration['access_operation'] ?? NULL,
      '#states' => [
        'visible' => [
          ':input[name="configuration[access_check]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $form;
  }

  /**
   * Handles bundle configuration portion of the form via AJAX.
   */
  public function updateBundlesAjax(array $form) {
    return $form['configuration']['bundles'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $bundles = array_keys(array_filter($form_state->getValue(['configuration', 'bundles'], [])));
    $form_state->setValue(['configuration', 'bundles'], $bundles);

    $entity_type = $form_state->getValue(['configuration', 'entity_type']);
    if ($entity_type != ($this->configuration['entity_type'] ?? NULL)) {
      $this->configuration['entity_type'] = $entity_type;
      $form_state->setRebuild();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    // No op.
  }

  /**
   * {@inheritdoc}
   */
  public function applySubscriptionData(AdvancedMessageSubscriptionEntity $subscription): bool {
    if (!$this->requiredDataAvailable() || !$this->checkBundle($subscription->getType())) {
      return FALSE;
    }

    $data = $this->getRequiredData();
    $subscription->set($this->configuration['field_name'], $data['entity']);
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getUserSubscription(AdvancedMessageSubscriptionType $subscription_type, ?AccountInterface $account = NULL, array $data = [], bool $active_only = TRUE): ?AdvancedMessageSubscriptionEntity {
    if (!$this->checkBundle($subscription_type)) {
      return NULL;
    }

    // Ensure we have an entity value, that it's a content entity, and that it
    // is the expected entity type.
    if ((!$entity = $data['entity'] ?? NULL) || (!$entity instanceof ContentEntityInterface)) {
      return NULL;
    }
    if ($entity->getEntityTypeId() != $this->configuration['entity_type']) {
      return NULL;
    }

    // Bundle filtering.
    if (!empty($this->configuration['bundles']) && !in_array($entity->bundle(), $this->configuration['bundles'])) {
      return NULL;
    }

    if (!$account) {
      $account = $this->currentUser();
    }

    $query = $this->entityTypeManager()->getStorage('advanced_message_subscription')
      ->getQuery();

    $owner_key = $this->entityTypeManager()->getDefinition('advanced_message_subscription')->getKey('owner');
    $target_entity_id_key = $this->entityTypeManager()->getDefinition($this->configuration['entity_type'])->getKey('id');
    $query->condition($owner_key, $account->id())
      ->condition('bundle', $subscription_type->id())
      ->condition($this->configuration['field_name'] . '.entity.' . $target_entity_id_key, $data['entity']->id());

    if ($active_only) {
      $query->condition('status', 1);
    }

    $result = $query->range(0, 1)->accessCheck(FALSE)->execute();
    if ($id = reset($result)) {
      return $this->entityTypeManager()->getStorage('advanced_message_subscription')->load($id);
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'field_name' => NULL,
      'entity_type' => NULL,
      'entity_bundles' => [],
      'extra_field_display' => TRUE,
      'access_check' => TRUE,
      'access_operation' => 'view',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function createAccess(AdvancedMessageSubscriptionType $subscription_type, ?DataProvider $dataProvider = NULL): AccessResultInterface {
    $access = parent::createAccess($subscription_type, $dataProvider);

    if (!$dataProvider) {
      $dataProvider = $this->dataProvider();
    }
    $data = $this->getRequiredData($dataProvider);
    $entity = $data['entity'] ?? NULL;

    if (!empty($this->configuration['access_check'])) {
      if ($entity) {
        $entity_access = $entity->access($this->configuration['access_operation'], $this->currentUser(), TRUE)
          ->addCacheableDependency($entity);
      }
      else {
        $entity_access = AccessResult::allowed();
      }
      $entity_access->addCacheableDependency($entity);
      $access = $access->andIf($entity_access);
    }

    return $access;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags(AdvancedMessageSubscriptionEntity $subscription): array {
    if (!$field_name = $this->configuration['field_name'] ?? NULL) {
      return parent::getCacheTags($subscription);
    }

    if (!$entity = $subscription->$field_name?->entity) {
      return parent::getCacheTags($subscription);
    }

    return Cache::mergeTags(parent::getCacheTags($subscription), $entity->getCacheTags());
  }

}
