<?php

declare(strict_types=1);

namespace Drupal\advanced_message_subscription\Form;

use Drupal\Core\Utility\Token;
use Drupal\advanced_message_subscription\Entity\AdvancedMessageSubscription;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form controller for the advanced message subscription entity edit forms.
 */
class AdvancedMessageSubscriptionForm extends ContentEntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $subscription = $this->entity;
    assert($subscription instanceof AdvancedMessageSubscription);
    $type = $subscription->getType();

    if ($subscription->isNew()) {
      $name = $this::getToken()->replace($type->getNamePattern(), ['advanced_message_subscription' => $subscription]);
      $subscription->name = $name;
    }

    return parent::form($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);

    $message_args = ['%label' => $this->entity->toLink()->toString()];
    $logger_args = [
      '%label' => $this->entity->label(),
      'link' => $this->entity->toLink($this->t('View'))->toString(),
    ];

    switch ($result) {
      case SAVED_NEW:
        $this->messenger()->addStatus($this->t('New advanced message subscription %label has been created.', $message_args));
        $this->logger('advanced_message_subscription')->notice('New advanced message subscription %label has been created.', $logger_args);
        break;

      case SAVED_UPDATED:
        $this->messenger()->addStatus($this->t('The advanced message subscription %label has been updated.', $message_args));
        $this->logger('advanced_message_subscription')->notice('The advanced message subscription %label has been updated.', $logger_args);
        break;

      default:
        throw new \LogicException('Could not save the entity.');
    }

    $form_state->setRedirectUrl($this->entity->toUrl('collection'));

    return $result;
  }

  /**
   * Get the token service.
   *
   * @return \Drupal\Core\Utility\Token
   *   The token service.
   */
  protected static function getToken() :Token {
    return \Drupal::token();
  }

}
