<?php

namespace Drupal\advanced_toast;

use Drupal\Core\Messenger\MessengerInterface;

/**
 * Service for managing toast notifications.
 */
class AdvancedToastService {

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The component plugin manager.
   *
   * @var mixed
   */
  protected $componentPluginManager;

  /**
   * Constructs a AdvancedToastService object.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param mixed $component_plugin_manager
   *   The component plugin manager.
   */
  public function __construct(
    MessengerInterface $messenger,
    $component_plugin_manager
  ) {
    $this->messenger = $messenger;
    $this->componentPluginManager = $component_plugin_manager;
  }

  /**
   * Add a toast notification.
   *
   * @param string $message
   *   The message to display.
   * @param string $type
   *   The toast type (status, warning, error, or custom).
   * @param array $options
   *   Additional options:
   *   - duration: Display duration in milliseconds (default: 5000).
   *   - dismissible: Whether the toast can be dismissed (default: TRUE).
   *   - fallback: Fallback toast type to use if type component not found.
   *
   * @return $this
   */
  public function addToast(string $message, string $type = 'status', array $options = []) {
    $config = \Drupal::config('advanced_toast.settings');

    // Determine the component to use from type mapping.
    $component_name = $this->getComponentForType($type);

    // Prepare toast data.
    $toast_data = [
      'type' => $type,
      'message' => $message,
      'duration' => $options['duration'] ?? $config->get('default_duration') ?? 5000,
      // Support both 'dismissible' and 'dismissable' spellings.
      'dismissible' => $options['dismissible'] ?? $options['dismissable'] ?? $config->get('default_dismissible') ?? TRUE,
      'component' => $component_name,
    ];

    // Add fallback type if specified.
    if (!empty($options['fallback'])) {
      $toast_data['fallback'] = $options['fallback'];
    }

    // Store in tempstore for retrieval on next page load.
    $tempstore = \Drupal::service('tempstore.private')->get('advanced_toast');
    $toasts = $tempstore->get('toasts') ?? [];
    $toasts[] = $toast_data;
    $tempstore->set('toasts', $toasts);

    return $this;
  }

  /**
   * Add a status toast.
   *
   * @param string $message
   *   The message to display.
   * @param array $options
   *   Additional options.
   *
   * @return $this
   */
  public function status(string $message, array $options = []) {
    return $this->addToast($message, 'status', $options);
  }

  /**
   * Add a warning toast.
   *
   * @param string $message
   *   The message to display.
   * @param array $options
   *   Additional options.
   *
   * @return $this
   */
  public function warning(string $message, array $options = []) {
    return $this->addToast($message, 'warning', $options);
  }

  /**
   * Add an error toast.
   *
   * @param string $message
   *   The message to display.
   * @param array $options
   *   Additional options.
   *
   * @return $this
   */
  public function error(string $message, array $options = []) {
    return $this->addToast($message, 'error', $options);
  }

  /**
   * Get the component name for a given toast type.
   *
   * @param string $type
   *   The toast type.
   *
   * @return string
   *   The component name.
   */
  protected function getComponentForType(string $type): string {
    $config = \Drupal::config('advanced_toast.settings');
    $type_mapping = $config->get('type_component_mapping') ?? [];

    // Default mappings - base components are in the module.
    $default_mapping = [
      'status' => 'advanced_toast:toast-status',
      'warning' => 'advanced_toast:toast-warning',
      'error' => 'advanced_toast:toast-error',
    ];

    $mapping = array_merge($default_mapping, $type_mapping);

    // If type not found in mapping, construct a component ID to check.
    // This allows custom types to be attempted before falling back.
    return $mapping[$type] ?? "advanced_toast:toast-{$type}";
  }

  /**
   * Resolve a component with fallback.
   *
   * Checks if the specified component exists. If not, falls back through:
   * 1. Original component for the type
   * 2. Component for the fallback type (if specified)
   * 3. Base toast component
   *
   * @param string $component_id
   *   The component ID to resolve.
   * @param string|null $fallback
   *   Optional fallback toast type to use instead of base toast.
   *
   * @return string
   *   The resolved component ID (or fallback).
   */
  public function resolveComponentWithFallback(string $component_id, ?string $fallback = NULL): string {
    // Check if the component exists using the component plugin manager.
    try {
      if ($this->componentPluginManager?->hasDefinition($component_id)) {
        return $component_id;
      }
    }
    catch (\Exception) {
      // Exception caught, will fall through to fallback logic.
    }

    // Component not found. Try the fallback type if specified.
    if ($fallback) {
      $fallback_component = $this->getComponentForType($fallback);

      // Check if the fallback component exists.
      try {
        if ($this->componentPluginManager?->hasDefinition($fallback_component)) {
          // Log that we're using the fallback.
          \Drupal::logger('advanced_toast')->warning('Toast component "@component" not found, falling back to "@fallback" type.', [
            '@component' => $component_id,
            '@fallback' => $fallback,
          ]);
          return $fallback_component;
        }
      }
      catch (\Exception) {
        // Fallback component also doesn't exist, will use base toast.
      }

      // Fallback component also doesn't exist.
      \Drupal::logger('advanced_toast')->warning('Toast component "@component" and fallback type "@fallback" not found, falling back to base toast.', [
        '@component' => $component_id,
        '@fallback' => $fallback,
      ]);
    }
    else {
      // No fallback specified, go straight to base toast.
      \Drupal::logger('advanced_toast')->warning('Toast component "@component" not found, falling back to base toast.', [
        '@component' => $component_id,
      ]);
    }

    return 'advanced_toast:toast';
  }

  /**
   * Get pending toasts from tempstore.
   *
   * @return array
   *   Array of toast data.
   */
  public function getPendingToasts(): array {
    $tempstore = \Drupal::service('tempstore.private')->get('advanced_toast');
    $toasts = $tempstore->get('toasts') ?? [];
    $tempstore->delete('toasts');
    return $toasts;
  }

}
