<?php

namespace Drupal\advanced_toast;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Security\TrustedCallbackInterface;

/**
 * Lazy builder service for rendering toast notifications.
 */
class AdvancedToastLazyBuilder implements TrustedCallbackInterface {

  /**
   * The toast service.
   *
   * @var \Drupal\advanced_toast\AdvancedToastServiceInterface
   */
  protected $toastService;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs an AdvancedToastLazyBuilder object.
   *
   * @param \Drupal\advanced_toast\AdvancedToastServiceInterface $toast_service
   *   The toast service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(
    AdvancedToastServiceInterface $toast_service,
    MessengerInterface $messenger,
    ConfigFactoryInterface $config_factory,
    RendererInterface $renderer
  ) {
    $this->toastService = $toast_service;
    $this->messenger = $messenger;
    $this->configFactory = $config_factory;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['renderToasts'];
  }

  /**
   * Lazy builder callback to render toast messages.
   *
   * @return array
   *   A render array containing the toast messages and attachments.
   */
  public function renderToasts() {
    $config = $this->configFactory->get('advanced_toast.settings');

    // Get toasts from tempstore (from PHP/Twig API).
    $toast_messages = $this->toastService->getPendingToasts();

    // Convert Drupal messages to toasts based on configuration.
    $replace_drupal_messages = $config->get('replace_drupal_messages') ?? FALSE;
    $type_mapping = $config->get('type_component_mapping') ?? [];

    // Map Drupal message types to toast types.
    $drupal_to_toast_map = [
      'status' => 'status',
      'warning' => 'warning',
      'error' => 'error',
    ];

    // Only convert messages if the global replace setting is enabled.
    if ($replace_drupal_messages) {
      foreach ($drupal_to_toast_map as $drupal_type => $toast_type) {
        $messages = $this->messenger->messagesByType($drupal_type);

        // Only convert if toast type has a component mapping.
        if (!empty($messages) && isset($type_mapping[$toast_type])) {
          foreach ($messages as $message) {
            $toast_messages[] = [
              'type' => $toast_type,
              'message' => (string) $message,
              'component' => $type_mapping[$toast_type],
            ];
          }
          // Delete messages only if they're being converted to toasts.
          $this->messenger->deleteByType($drupal_type);
        }
      }
    }

    // If no toasts, return empty.
    if (empty($toast_messages)) {
      return [];
    }

    // Build render array with rendered toasts and attachments.
    $build = [
      '#attached' => [
        'drupalSettings' => [
          'advancedToast' => [
            'pendingToasts' => [],
          ],
        ],
      ],
    ];

    foreach ($toast_messages as $toast) {
      $component_id = $toast['component'] ?? 'advanced_toast:toast-status';
      $fallback = $toast['fallback'] ?? NULL;

      // Resolve component with fallback to ensure it exists.
      $component_id = $this->toastService->resolveComponentWithFallback($component_id, $fallback);

      // Build the component as a render array.
      $component_build = [
        '#type' => 'component',
        '#component' => $component_id,
        '#props' => [
          'message' => $toast['message'],
          'dismissible' => (bool) ($toast['dismissible'] ?? TRUE),
        ],
      ];

      // Render the component using render() - not renderRoot() which can't be
      // used inside a lazy builder context.
      $rendered = $this->renderer->render($component_build);

      // Attach the SDC component's auto-generated library.
      $library_name = str_replace(':', '/', $component_id);
      $build['#attached']['library'][] = $library_name;

      // Pass rendered HTML and metadata to JavaScript.
      $build['#attached']['drupalSettings']['advancedToast']['pendingToasts'][] = [
        'html' => trim((string) $rendered),
        'type' => $toast['type'],
        'duration' => $toast['duration'] ?? NULL,
      ];
    }

    return $build;
  }

}
