<?php

namespace Drupal\advanced_toast;

/**
 * Interface for the advanced toast service.
 */
interface AdvancedToastServiceInterface {

  /**
   * Add a toast notification.
   *
   * @param string $message
   *   The message to display.
   * @param string $type
   *   The toast type (status, warning, error, or custom).
   * @param array $options
   *   Additional options:
   *   - duration: Display duration in milliseconds (default: 5000).
   *   - dismissible: Whether the toast can be dismissed (default: TRUE).
   *   - fallback: Fallback toast type to use if type component not found.
   *
   * @return $this
   */
  public function addToast(string $message, string $type = 'status', array $options = []);

  /**
   * Add a status toast.
   *
   * @param string $message
   *   The message to display.
   * @param array $options
   *   Additional options.
   *
   * @return $this
   */
  public function status(string $message, array $options = []);

  /**
   * Add a warning toast.
   *
   * @param string $message
   *   The message to display.
   * @param array $options
   *   Additional options.
   *
   * @return $this
   */
  public function warning(string $message, array $options = []);

  /**
   * Add an error toast.
   *
   * @param string $message
   *   The message to display.
   * @param array $options
   *   Additional options.
   *
   * @return $this
   */
  public function error(string $message, array $options = []);

  /**
   * Resolve a component with fallback.
   *
   * Checks if the specified component exists. If not, falls back through:
   * 1. Original component for the type
   * 2. Component for the fallback type (if specified)
   * 3. Base toast component
   *
   * @param string $component_id
   *   The component ID to resolve.
   * @param string|null $fallback
   *   Optional fallback toast type to use instead of base toast.
   *
   * @return string
   *   The resolved component ID (or fallback).
   */
  public function resolveComponentWithFallback(string $component_id, ?string $fallback = NULL): string;

  /**
   * Get pending toasts from tempstore.
   *
   * @return array
   *   Array of toast data.
   */
  public function getPendingToasts(): array;

}
