<?php

namespace Drupal\advanced_toast\Controller;

use Drupal\advanced_toast\Ajax\AddToastCommand;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for toast AJAX callbacks.
 */
class ToastController extends ControllerBase {

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The advanced toast service.
   *
   * @var \Drupal\advanced_toast\AdvancedToastService
   */
  protected $toastService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->renderer = $container->get('renderer');
    $instance->toastService = $container->get('advanced_toast.toast');
    return $instance;
  }

  /**
   * Render toast components via AJAX.
   *
   * This endpoint is called by JavaScript to get pending toasts
   * and returns an AjaxResponse with commands to display them.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   AJAX response with commands to display toasts.
   */
  public function renderToasts() {
    $response = new AjaxResponse();

    // Get pending toasts from the service.
    $toast_service = \Drupal::service('advanced_toast.toast');
    $toast_messages = $toast_service->getPendingToasts();

    if (empty($toast_messages)) {
      return $response;
    }

    // Render each toast and add as a command.
    foreach ($toast_messages as $toast) {
      $component_id = $toast['component'] ?? 'advanced_toast:toast-status';

      // Resolve component with fallback to ensure it exists.
      $component_id = $this->toastService->resolveComponentWithFallback($component_id);

      // Build the component as a render array.
      $build = [
        '#type' => 'component',
        '#component' => $component_id,
        '#props' => [
          'message' => $toast['message'],
          'dismissible' => $toast['dismissible'] ?? TRUE,
        ],
      ];

      // Render the component - this will attach libraries automatically.
      $rendered = $this->renderer->renderRoot($build);

      // Collect attachments from the render array.
      $attachments = $build['#attached'] ?? [];

      // Add command with rendered HTML and metadata.
      $toast_data = [
        'type' => $toast['type'],
        'duration' => $toast['duration'] ?? NULL,
      ];

      $response->addCommand(new AddToastCommand((string) $rendered, $toast_data));

      // Add attachments to the response.
      if (!empty($attachments)) {
        $existing_attachments = $response->getAttachments();
        $merged_attachments = array_merge_recursive($existing_attachments, $attachments);
        $response->setAttachments($merged_attachments);
      }
    }

    return $response;
  }

  /**
   * Render a single toast component via AJAX.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   AJAX response with command to display toast.
   */
  public function renderToast(Request $request) {
    $response = new AjaxResponse();

    $component_id = $request->query->get('component', 'advanced_toast:toast-status');
    $message = $request->query->get('message', '');
    $dismissible = $request->query->get('dismissible', 'true') === 'true';
    $type = $request->query->get('type', 'status');
    $duration = $request->query->get('duration');

    // Resolve component with fallback to ensure it exists.
    $component_id = $this->toastService->resolveComponentWithFallback($component_id);

    // Build the component as a render array.
    $build = [
      '#type' => 'component',
      '#component' => $component_id,
      '#props' => [
        'message' => $message,
        'dismissible' => $dismissible,
      ],
    ];

    // Render the component - this will attach libraries automatically.
    $rendered = $this->renderer->renderRoot($build);

    // Collect attachments from the render array.
    $attachments = $build['#attached'] ?? [];

    // Add command with rendered HTML and metadata.
    $toast_data = [
      'type' => $type,
      'duration' => $duration !== NULL ? (int) $duration : NULL,
    ];

    $response->addCommand(new AddToastCommand((string) $rendered, $toast_data));

    // Add attachments to the response.
    if (!empty($attachments)) {
      $response->setAttachments($attachments);
    }

    return $response;
  }

}
