# Advanced Toast Messages

A modern, flexible toast notification system for Drupal 10.3+ and Drupal 11 built on Single Directory Components (SDC).

## Overview

Traditional Drupal messages appear at the top of the page, requiring users to scroll up to see feedback after form submissions or actions. Toast notifications solve this by displaying temporary, non-intrusive messages that appear in a fixed position (like bottom-right corner), providing immediate feedback without disrupting the user's workflow.

This module was created to provide a **component-based** toast system that:
- Integrates seamlessly with Drupal's SDC architecture
- Allows complete theming control without touching module code
- Supports extending with custom toast types per theme
- Provides multiple APIs (PHP, Twig, JavaScript) for maximum flexibility
- Works alongside or replaces standard Drupal messages

## Features

- **Module-based base components** - Status, warning, and error toasts included in the module
- **Theme override support** - Override styles in your theme for custom branding
- **Extensible** - Add custom toast types in themes or other modules
- **PHP, Twig & JavaScript API** - Trigger toasts from PHP services, Twig templates, or JavaScript
- **Drupal message replacement** - Optionally convert Drupal messages to toasts globally
- **Configurable** - Admin UI for toast behavior and component mappings
- **Three-tier fallback system** - Ensures toasts always render even if components are missing
- **Accessible** - ARIA live regions, keyboard support, and respects reduced motion preferences

## Requirements

- Drupal 10.3+ or Drupal 11+
- Single Directory Components (SDC) module (core)

## Installation

```bash
drush en advanced_toast
drush cr
```

## Architecture

### Component Resolution

Toast components are resolved using a configurable type-to-component mapping system. Configure mappings at `/admin/config/user-interface/toast`.

**Three-Tier Fallback System:**

1. **Primary**: Renders the component mapped to the specified type
2. **Fallback**: If primary doesn't exist and a fallback type is provided, tries that component
3. **Base**: If both fail, renders the base `advanced_toast:toast` component

**Example Flow:**
```php
$toast->addToast('Custom info message', 'info', ['fallback' => 'status']);
```

1. System checks type mapping for 'info' → `us_hub:toast-info` (if mapped in config)
2. If `us_hub:toast-info` doesn't exist, tries 'status' → `advanced_toast:toast-status`
3. If that doesn't exist, uses `advanced_toast:toast`
4. Logs appropriate warning at each fallback level for admin visibility

### Base Components (Module)

The module includes these base components:

- `advanced_toast:toast` - Base toast component (minimal fallback, no icon)
- `advanced_toast:toast-status` - Status notifications (green checkmark icon)
- `advanced_toast:toast-warning` - Warning notifications (yellow triangle icon)
- `advanced_toast:toast-error` - Error notifications (red X icon)

All components support:
- `message` (required): The message text to display
- `dismissible` (optional): Whether to show close button (default: true)
- `utility_classes` (optional): Additional CSS classes
- `type` (optional): Toast type for styling

### Theme Customization

- Override component styles with theme CSS
- Add custom toast types (e.g., `us_hub:toast-info`, `us_hub:toast-celebration`)
- Map custom types in admin UI to use theme components
- Replace base components completely if needed
- Use third-party libraries (e.g., js-confetti) for enhanced visual effects

## Usage

### From PHP

```php
$toast = \Drupal::service('advanced_toast.toast');

// Standard toasts using shorthand methods
$toast->status('Your changes have been saved!');
$toast->warning('Please review your submission.');
$toast->error('An error occurred. Please try again.');

// Custom toast with options
$toast->addToast('Custom message', 'info', [
  'duration' => 3000,           // Display duration in milliseconds
  'dismissible' => FALSE,       // Hide close button
  'fallback' => 'status',       // Fallback to status if info component not found
]);

// Theme-specific toast (e.g., celebration)
$toast->addToast('🎉 Congratulations!', 'celebration', [
  'fallback' => 'status',
]);
```

### From Twig

```twig
{# Standard toasts #}
{{ toast_status('Your changes have been saved!') }}
{{ toast_warning('Please review your submission.') }}
{{ toast_error('An error occurred.') }}

{# Custom toast with options #}
{{ toast('Custom message', 'info', {
  duration: 3000,
  dismissible: false,
  fallback: 'status'
}) }}

{# Theme-specific toast #}
{{ toast('🎉 Congratulations!', 'celebration', {
  fallback: 'status'
}) }}
```

### From JavaScript

```javascript
// Standard toasts
Drupal.toast('Status message', 'status');
Drupal.toast('Warning message', 'warning');
Drupal.toast('Error occurred', 'error');

// Custom toast with options
Drupal.toast('Custom message', 'info', {
  duration: 3000,
  dismissible: false,  // or 'dismissable'
  fallback: 'status'   // Falls back to status if info component not found
});

// Theme-specific toast
Drupal.toast('🎉 Congratulations!', 'celebration', {
  fallback: 'status'
});
```

## Configuration

Visit `/admin/config/user-interface/toast` to configure:

### Global Settings

- **Replace Drupal messages with toasts** - Single toggle to convert all Drupal messages to toasts (only converts types that have component mappings)
- **Default toast duration** - Display time in milliseconds (default: 5000)
- **Toasts dismissible by default** - Whether toasts show a close button by default (default: true)
- **Toast position** - Screen position: top-right, top-left, bottom-right, bottom-left, top-center, bottom-center

### Component Mappings

Map toast types to SDC components using a dynamic table:

- **Base types** (status, warning, error) are pre-populated
- **Override base components** by changing the Component ID field
- **Add custom types** in empty rows (e.g., info, celebration, success)
- **Add more rows** button to expand the table
- **Leave rows empty** to ignore them

**Important:** Only toast types with component mappings will be used when converting Drupal messages. For example:
- If only `status` is mapped, warning and error messages remain as regular Drupal messages
- If all three base types are mapped, all standard Drupal messages convert to toasts
- Custom types work independently of Drupal message replacement

### Component Validation

The form validates that:
- Component IDs are valid and exist in the system
- Components are marked as toast components via `thirdPartySettings.advanced_toast.is_toast_component: true`
- Components have the required `message` property of type `string`

## Customization

### Method 1: Override Base Styles

**1. Create override CSS in your theme:**

```css
/* themes/custom/us_hub/css/toast-overrides.css */
.advanced-toast--status {
  border-left-color: #00ff00; /* Your brand color */
}

.advanced-toast--status .advanced-toast__icon {
  color: #00ff00;
}
```

**2. Extend the toast library in `us_hub.info.yml`:**

```yaml
libraries-extend:
  advanced_toast/toast:
    - us_hub/toast-overrides
```

**3. Define library in `us_hub.libraries.yml`:**

```yaml
toast-overrides:
  css:
    theme:
      css/toast-overrides.css: {}
```

### Method 2: Create Custom Toast Type

This example creates a celebration toast using the js-confetti library.

**1. Create component directory:**

```
themes/custom/us_hub/components/toast-celebration/
├── toast-celebration.component.yml
├── toast-celebration.twig
├── toast-celebration.css
└── toast-celebration.js
```

**2. Define component (`toast-celebration.component.yml`):**

```yaml
$schema: https://git.drupalcode.org/project/drupal/-/raw/10.1.x/core/modules/sdc/src/metadata.schema.json
name: Toast Celebration
status: stable
description: Celebration toast notification with confetti animation
libraryOverrides:
  dependencies:
    - us_hub/js-confetti
thirdPartySettings:
  advanced_toast:
    is_toast_component: true
props:
  type: object
  properties:
    message:
      type: string
      title: Message
      description: The celebration message to display
    dismissible:
      type: boolean
      title: Dismissible
      default: true
    confetti_duration:
      type: number
      title: Confetti Duration
      default: 3000
    utility_classes:
      type: array
      title: Utility Classes
    attributes:
      type: Drupal\Core\Template\Attribute
      title: HTML Attributes
  required:
    - message
```

**3. Create template (`toast-celebration.twig`):**

```twig
{# Add celebration-specific class to utility classes #}
{% set celebration_classes = utility_classes|default([])|merge(['toast--celebration']) %}

{# Render using the base toast component with status type #}
{% set celebration_attributes = attributes.setAttribute('data-confetti-duration', confetti_duration|default(3000)) %}

{% include 'advanced_toast:toast' with {
  message: message,
  type: 'status',
  dismissible: dismissible,
  utility_classes: celebration_classes,
  attributes: celebration_attributes,
} %}
```

**4. Add styles (`toast-celebration.css`):**

```css
/* Override status toast background for celebration */
.advanced-toast--status.toast--celebration {
  background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
  border-left-color: #f59e0b;
  position: relative;
  overflow: visible;
}

/* Confetti canvas positioning - extends beyond toast boundaries */
.toast-celebration__confetti {
  position: absolute;
  top: -100px;
  left: -100px;
  width: calc(100% + 200px);
  height: calc(100% + 200px);
  pointer-events: none;
  z-index: 1;
}
```

**5. Add confetti behavior (`toast-celebration.js`):**

```javascript
(function (Drupal, once) {
  'use strict';

  Drupal.behaviors.toastCelebrationConfetti = {
    attach: function (context) {
      once('toast-celebration-confetti', '.advanced-toast--status.toast--celebration', context).forEach(function (toast) {
        if (typeof JSConfetti === 'undefined') {
          return;
        }

        const canvas = document.createElement('canvas');
        canvas.className = 'toast-celebration__confetti';
        toast.appendChild(canvas);

        const jsConfetti = new JSConfetti({ canvas });
        jsConfetti.addConfetti({
          confettiRadius: 6,
          confettiNumber: 150,
          confettiColors: ['#FF6B6B', '#4ECDC4', '#45B7D1', '#FFA07A']
        });
      });
    }
  };
})(Drupal, once);
```

**6. Install js-confetti library:**

```bash
cd themes/custom/us_hub
npm install js-confetti --save
```

**7. Add library definition (`us_hub.libraries.yml`):**

```yaml
js-confetti:
  js:
    node_modules/js-confetti/dist/js-confetti.browser.js: { minified: true }
```

**8. Map toast type in admin UI:**

Go to `/admin/config/user-interface/toast` and add:
- Toast Type: `celebration`
- Component ID: `us_hub:toast-celebration`

**9. Use it:**

```php
\Drupal::service('advanced_toast.toast')->addToast('🎉 Achievement unlocked!', 'celebration');
```

## Component Structure

### Module Components

The module provides base components for standard toast types:

```
modules/custom/advanced_toast/components/
├── toast/                    # Minimal fallback component (no icon)
│   ├── toast.component.yml
│   ├── toast.twig
│   └── toast.css
├── toast-status/            # Status toast with checkmark icon
│   ├── toast-status.component.yml
│   ├── toast-status.twig
│   └── toast-status.css
├── toast-warning/           # Warning toast with triangle icon
│   └── ...
└── toast-error/             # Error toast with X icon
    └── ...
```

### Theme Components

Create custom toast types in your theme:

```
themes/custom/us_hub/components/toast-[name]/
├── toast-[name].component.yml
├── toast-[name].twig
├── toast-[name].css
└── toast-[name].js  # Optional - for interactive features
```

**Component Requirements:**

All toast components must:
1. Define `thirdPartySettings.advanced_toast.is_toast_component: true` in the YAML
2. Have a `message` property of type `string` in props

**Template Inheritance:**

Components can use the base toast template via `{% include %}`:

```twig
{% include 'advanced_toast:toast' with {
  message: message,
  type: 'status',
  dismissible: dismissible,
  utility_classes: utility_classes|default([])|merge(['custom-class']),
} %}
```

## Accessibility

- **ARIA live regions** - `aria-live="polite"` for status/warning, `aria-live="assertive"` for errors
- **Proper ARIA labels** - Dismiss buttons have `aria-label="Dismiss"`
- **Keyboard accessible** - All interactive elements are keyboard navigable
- **Reduced motion** - Respects `prefers-reduced-motion` media query
- **Screen reader friendly** - Toast content is announced to screen readers

## Troubleshooting

### Toasts not appearing

```bash
drush en advanced_toast sdc
drush cr
```

Check browser console for JavaScript errors.

### Service error

Requires Drupal 10.3+ or Drupal 11+ with SDC support. Ensure the SDC module is enabled.

### Custom components not working

```bash
drush cr  # Clear cache after adding components
```

Verify component validation:
- Check `thirdPartySettings.advanced_toast.is_toast_component: true` is set
- Ensure `message` property exists and is type `string`

### Component validation errors on settings form

The form validates components and shows inline errors:
- **"Component ... does not exist"** - Component ID is incorrect or component not found
- **"Component ... is not a toast component"** - Missing `thirdPartySettings.advanced_toast.is_toast_component: true`
- **"Component ... does not have required 'message' property"** - Add message property to props

### Styles not applying

- Check library extends in theme `.info.yml` file
- Verify CSS specificity (module uses `.advanced-toast--[type]`)
- Clear cache with `drush cr`
- Check browser developer tools for CSS conflicts

### JavaScript API not working

Ensure the toast library is loaded:
```javascript
if (typeof Drupal.toast !== 'undefined') {
  Drupal.toast('Message', 'status');
}
```

### Confetti or animations not working in theme components

- Verify library dependencies are listed in `libraryOverrides.dependencies`
- Check that npm packages are installed
- Ensure canvas has `overflow: visible` on parent elements
- Check browser console for JavaScript errors

## API Reference

### PHP Service Methods

```php
$toast = \Drupal::service('advanced_toast.toast');

// Shorthand methods
$toast->status(string $message, array $options = []);
$toast->warning(string $message, array $options = []);
$toast->error(string $message, array $options = []);

// General method
$toast->addToast(string $message, string $type = 'status', array $options = []);
```

**Options:**
- `duration` (int): Display duration in milliseconds
- `dismissible` (bool): Whether to show close button
- `fallback` (string): Fallback toast type if primary component not found

### Twig Functions

```twig
{{ toast_status(message, options = {}) }}
{{ toast_warning(message, options = {}) }}
{{ toast_error(message, options = {}) }}
{{ toast(message, type = 'status', options = {}) }}
```

### JavaScript API

```javascript
Drupal.toast(message, type = 'status', options = {});
```

**Options:**
- `duration` (number): Display duration in milliseconds
- `dismissible` (boolean): Whether to show close button
- `fallback` (string): Fallback toast type

## Technical Details

### Component Discovery

Components are discovered via Drupal's SDC plugin system. The module:
1. Scans all enabled modules and themes for SDC components
2. Checks `thirdPartySettings.advanced_toast.is_toast_component`
3. Validates component schema for required `message` property
4. Makes components available via component ID (e.g., `us_hub:toast-celebration`)

### Rendering Pipeline

1. **PHP/Twig API** → Stores toast in private tempstore
2. **hook_page_attachments()** → Retrieves toasts from tempstore
3. **Component resolution** → Resolves component via type mapping with fallback
4. **Server-side rendering** → Renders component to HTML with attachments
5. **JavaScript injection** → Passes HTML + metadata to JavaScript via drupalSettings
6. **Client-side display** → JavaScript creates toast container and displays toasts

### Drupal Message Replacement

When enabled via the "Replace Drupal messages with toasts" setting:
1. Module reads Drupal messages from messenger service
2. Checks if each message type has a component mapping
3. Converts mapped types to toasts
4. Deletes original Drupal messages (only for converted types)
5. Non-mapped types remain as standard Drupal messages

## Performance Considerations

- **Server-side rendering** - Components are rendered server-side for SEO and accessibility
- **Attachment collection** - Component CSS/JS libraries are automatically collected
- **Tempstore cleanup** - Toast messages are automatically cleared after rendering
- **Canvas optimization** - Confetti animations use requestAnimationFrame for smooth performance
- **Reduced motion** - Animations automatically disabled when user prefers reduced motion

**Co-authored with [Claude Code](https://claude.com/claude-code)** - AI pair programming assistant by Anthropic.
