/**
 * @file
 * Toast notification manager.
 */

(function (Drupal, drupalSettings) {
  'use strict';

  /**
   * Toast Manager class.
   */
  class ToastManager {
    constructor() {
      this.container = null;
      this.toasts = new Map();
      this.settings = drupalSettings.advancedToast || {};
      this.init();
    }

    /**
     * Initialize the toast manager.
     */
    init() {
      // Create toast container
      this.createContainer();

      // Process pending toasts from drupalSettings
      if (this.settings.pendingToasts && Array.isArray(this.settings.pendingToasts)) {
        this.settings.pendingToasts.forEach(toast => {
          this.addToast(toast.html, {
            type: toast.type,
            duration: toast.duration
          });
        });
      }
    }

    /**
     * Create the toast container element.
     */
    createContainer() {
      const position = this.settings.position || 'top-right';
      this.container = document.createElement('div');
      this.container.className = `advanced-toast-container advanced-toast-container--${position}`;
      this.container.setAttribute('aria-live', 'polite');
      this.container.setAttribute('aria-atomic', 'false');
      document.body.appendChild(this.container);
    }

    /**
     * Add a toast notification to the DOM.
     *
     * This is called by the custom AJAX command from the server.
     *
     * @param {string} html - The rendered toast HTML.
     * @param {Object} toastData - Toast metadata (type, duration).
     */
    addToast(html, toastData = {}) {
      const toastId = `toast-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
      const duration = toastData.duration !== undefined && toastData.duration !== null
        ? toastData.duration
        : (this.settings.defaultDuration !== undefined ? this.settings.defaultDuration : 5000);

      // Parse HTML and create element
      const wrapper = document.createElement('div');
      wrapper.innerHTML = html;
      const toastElement = wrapper.firstElementChild;

      if (!toastElement) {
        console.error('Failed to create toast element from HTML');
        return null;
      }

      toastElement.setAttribute('data-toast-id', toastId);

      // Add to container
      this.container.appendChild(toastElement);

      // Attach Drupal behaviors to the new toast element
      if (Drupal.attachBehaviors) {
        Drupal.attachBehaviors(toastElement);
      }

      // Store toast data
      this.toasts.set(toastId, {
        element: toastElement,
        timer: null,
        duration: duration
      });

      // Trigger entrance animation
      requestAnimationFrame(() => {
        toastElement.classList.add('advanced-toast--visible');
      });

      // Set up auto-dismiss (only if duration > 0)
      if (duration > 0) {
        this.startProgressBar(toastId, duration);
        const timer = setTimeout(() => {
          this.dismiss(toastId);
        }, duration);
        this.toasts.get(toastId).timer = timer;
      }

      // Set up dismiss button
      const dismissBtn = toastElement.querySelector('.advanced-toast__dismiss');
      if (dismissBtn) {
        dismissBtn.addEventListener('click', () => {
          this.dismiss(toastId);
        });
      }

      return toastId;
    }

    /**
     * Show a toast notification (manual API).
     *
     * @param {string} message - The message to display.
     * @param {string} type - The toast type (status, warning, error, or custom).
     * @param {Object} options - Additional options:
     *   - duration: Display duration in milliseconds (default: config)
     *   - dismissible/dismissable: Whether toast can be dismissed (default: true)
     *   - fallback: Fallback toast type if primary doesn't exist
     */
    show(message, type = 'status', options = {}) {
      // Support both 'dismissible' and 'dismissable' spellings
      const dismissible = options.dismissible !== undefined
        ? options.dismissible
        : (options.dismissable !== undefined ? options.dismissable : true);
      const duration = options.duration;
      const fallback = options.fallback;

      // Get component ID from type mapping or construct one
      const typeMapping = this.settings.typeComponentMapping || {};
      const component = typeMapping[type] || `advanced_toast:toast-${type}`;

      // Build query parameters
      const params = new URLSearchParams({
        message: message,
        type: type,
        component: component,
        dismissible: dismissible ? 'true' : 'false'
      });

      if (duration !== undefined) {
        params.append('duration', duration.toString());
      }

      if (fallback !== undefined) {
        params.append('fallback', fallback);
      }

      // Use Drupal AJAX to ensure commands are processed
      const ajax = Drupal.ajax({
        url: `/advanced-toast/render?${params}`,
        element: document.body
      });
      ajax.execute();
    }

    /**
     * Start the progress bar animation.
     *
     * @param {string} toastId - The toast ID.
     * @param {number} duration - Duration in milliseconds.
     */
    startProgressBar(toastId, duration) {
      const toast = this.toasts.get(toastId);
      if (!toast) return;

      const progressBar = toast.element.querySelector('.advanced-toast__progress');
      if (progressBar) {
        progressBar.style.animationDuration = `${duration}ms`;
        progressBar.classList.add('advanced-toast__progress--active');
      }
    }

    /**
     * Dismiss a toast notification.
     *
     * @param {string} toastId - The toast ID.
     */
    dismiss(toastId) {
      const toast = this.toasts.get(toastId);
      if (!toast) return;

      // Clear timer
      if (toast.timer) {
        clearTimeout(toast.timer);
      }

      // Trigger exit animation
      toast.element.classList.add('advanced-toast--hiding');
      toast.element.classList.remove('advanced-toast--visible');

      // Remove from DOM after animation
      setTimeout(() => {
        if (toast.element.parentNode) {
          toast.element.parentNode.removeChild(toast.element);
        }
        this.toasts.delete(toastId);
      }, 300);
    }

    /**
     * Dismiss all toasts.
     */
    dismissAll() {
      const toastIds = Array.from(this.toasts.keys());
      toastIds.forEach(id => this.dismiss(id));
    }
  }

  /**
   * Drupal behavior for toast notifications.
   */
  Drupal.behaviors.advancedToast = {
    attach: function (context, settings) {
      if (context !== document) return;

      // Initialize toast manager once
      if (!Drupal.toastManager) {
        Drupal.toastManager = new ToastManager();
      }
    }
  };

  /**
   * Custom AJAX command to add a toast.
   */
  Drupal.AjaxCommands.prototype.addToast = function (ajax, response) {
    if (Drupal.toastManager) {
      Drupal.toastManager.addToast(response.html, response.toastData);
    }
  };

  /**
   * Expose toast API.
   */
  Drupal.toast = function (message, type, options) {
    if (Drupal.toastManager) {
      return Drupal.toastManager.show(message, type, options);
    }
  };

})(Drupal, drupalSettings);
