<?php

namespace Drupal\advanced_toast\Ajax;

use Drupal\Core\Ajax\CommandInterface;
use Drupal\Core\Ajax\CommandWithAttachedAssetsInterface;
use Drupal\Core\Asset\AttachedAssets;

/**
 * AJAX command for adding a toast notification.
 *
 * This command handles rendering the toast component internally based on the
 * provided type and options, simplifying usage from controllers.
 */
class AddToastCommand implements CommandInterface, CommandWithAttachedAssetsInterface {

  /**
   * The toast message.
   *
   * @var string
   */
  protected $message;

  /**
   * The toast type.
   *
   * @var string
   */
  protected $type;

  /**
   * Toast options.
   *
   * @var array
   */
  protected $options;

  /**
   * The rendered HTML.
   *
   * @var string
   */
  protected $html;

  /**
   * The attached assets from rendering.
   *
   * @var array
   */
  protected $attachedLibraries = [];

  /**
   * Constructs an AddToastCommand object.
   *
   * @param string $message
   *   The toast message to display.
   * @param string $type
   *   The toast type (status, warning, error, coins, etc.).
   * @param array $options
   *   Additional options:
   *   - duration: Display duration in milliseconds (default: 5000).
   *   - dismissible: Whether the toast can be dismissed (default: TRUE).
   *   - fallback: Fallback toast type if component not found.
   *   - additional_props: Additional props to pass to the component.
   */
  public function __construct(string $message, string $type = 'status', array $options = []) {
    $this->message = $message;
    $this->type = $type;
    $this->options = $options;

    // Pre-render the component to capture HTML and attachments.
    $this->preRender();
  }

  /**
   * Pre-render the toast component to capture HTML and attachments.
   */
  protected function preRender(): void {
    $toast_service = \Drupal::service('advanced_toast.toast');
    $config = \Drupal::config('advanced_toast.settings');
    $renderer = \Drupal::service('renderer');

    // Get component ID based on type with fallback support.
    $component_id = $this->getComponentForType($this->type);
    $fallback = $this->options['fallback'] ?? NULL;
    $resolved_component = $toast_service->resolveComponentWithFallback($component_id, $fallback);

    // Build base props.
    $props = [
      'message' => $this->message,
      'dismissible' => $this->options['dismissible'] ?? $this->options['dismissable'] ?? $config->get('default_dismissible') ?? TRUE,
    ];

    // Merge additional props if provided.
    if (!empty($this->options['additional_props']) && is_array($this->options['additional_props'])) {
      $props = array_merge($props, $this->options['additional_props']);
    }

    // Build the component render array.
    $build = [
      '#type' => 'component',
      '#component' => $resolved_component,
      '#props' => $props,
    ];

    // Render the component and capture attachments.
    $this->html = (string) $renderer->renderPlain($build);

    // Capture attached libraries for getAttachedAssets().
    $this->attachedLibraries = $build['#attached']['library'] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $config = \Drupal::config('advanced_toast.settings');

    // Prepare toast metadata.
    $toast_data = [
      'type' => $this->type,
      'duration' => $this->options['duration'] ?? $config->get('default_duration') ?? 5000,
    ];

    return [
      'command' => 'addToast',
      'html' => $this->html,
      'toastData' => $toast_data,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAttachedAssets() {
    $assets = new AttachedAssets();
    $assets->setLibraries($this->attachedLibraries);
    return $assets;
  }

  /**
   * Get the component ID for a given toast type from config mapping.
   *
   * @param string $type
   *   The toast type.
   *
   * @return string
   *   The component ID from config, or base toast as fallback.
   */
  protected function getComponentForType(string $type): string {
    $config = \Drupal::config('advanced_toast.settings');
    $type_mapping = $config->get('type_component_mapping') ?? [];

    // Return from config mapping, or base toast as fallback.
    return $type_mapping[$type] ?? 'advanced_toast:toast';
  }

}
