<?php

namespace Drupal\advanced_toast\Controller;

use Drupal\advanced_toast\Ajax\AddToastCommand;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for toast AJAX callbacks.
 */
class ToastController extends ControllerBase {

  /**
   * Render toast components via AJAX.
   *
   * This endpoint is called by JavaScript to get pending toasts
   * and returns an AjaxResponse with commands to display them.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   AJAX response with commands to display toasts.
   */
  public function renderToasts() {
    $response = new AjaxResponse();

    // Get pending toasts from the service.
    $toast_service = \Drupal::service('advanced_toast.toast');
    $toast_messages = $toast_service->getPendingToasts();

    if (empty($toast_messages)) {
      return $response;
    }

    // Add each toast as a command. AddToastCommand handles rendering internally.
    foreach ($toast_messages as $toast) {
      $options = [
        'duration' => $toast['duration'] ?? NULL,
        'dismissible' => $toast['dismissible'] ?? TRUE,
      ];

      // Pass additional props if provided.
      if (!empty($toast['additional_props']) && is_array($toast['additional_props'])) {
        $options['additional_props'] = $toast['additional_props'];
      }

      $response->addCommand(new AddToastCommand(
        $toast['message'],
        $toast['type'] ?? 'status',
        $options
      ));
    }

    return $response;
  }

  /**
   * Render a single toast component via AJAX.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   AJAX response with command to display toast.
   */
  public function renderToast(Request $request) {
    $response = new AjaxResponse();

    $message = $request->query->get('message', '');
    $type = $request->query->get('type', 'status');
    $dismissible = $request->query->get('dismissible', 'true') === 'true';
    $duration = $request->query->get('duration');
    $additional_props = $request->query->get('additional_props');

    // Build options array.
    $options = [
      'dismissible' => $dismissible,
    ];

    if ($duration !== NULL) {
      $options['duration'] = (int) $duration;
    }

    // Decode and pass additional props if provided.
    if ($additional_props) {
      $decoded_props = json_decode($additional_props, TRUE);
      if (is_array($decoded_props)) {
        $options['additional_props'] = $decoded_props;
      }
    }

    // AddToastCommand handles rendering internally.
    $response->addCommand(new AddToastCommand($message, $type, $options));

    return $response;
  }

}
