<?php

namespace Drupal\advanced_toast\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Theme\ComponentPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Configure Advanced Toast Messages settings.
 */
class AdvancedToastSettingsForm extends ConfigFormBase {

  /**
   * The component plugin manager.
   *
   * @var \Drupal\Core\Theme\ComponentPluginManager
   */
  protected $componentPluginManager;

  /**
   * Constructs a AdvancedToastSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Theme\ComponentPluginManager $component_plugin_manager
   *   The component plugin manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, ComponentPluginManager $component_plugin_manager) {
    parent::__construct($config_factory);
    $this->componentPluginManager = $component_plugin_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('plugin.manager.sdc')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['advanced_toast.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'advanced_toast_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('advanced_toast.settings');

    // Attach admin CSS.
    $form['#attached']['library'][] = 'advanced_toast/admin';

    $form['replace_drupal_messages'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Replace Drupal messages with toasts'),
      '#description' => $this->t('When enabled, standard Drupal status, warning, and error messages will be converted to toast notifications.'),
      '#default_value' => $config->get('replace_drupal_messages') ?? FALSE,
    ];

    $form['default_duration'] = [
      '#type' => 'number',
      '#title' => $this->t('Default toast duration'),
      '#description' => $this->t('Duration in milliseconds that toasts will be displayed. Set to 0 for permanent toasts.'),
      '#default_value' => $config->get('default_duration') ?? 5000,
      '#min' => 0,
      '#step' => 100,
      '#field_suffix' => 'ms',
    ];

    $form['default_dismissible'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Toasts dismissible by default'),
      '#description' => $this->t('Whether toasts can be dismissed by clicking a close button by default. This can be overridden per toast.'),
      '#default_value' => $config->get('default_dismissible') ?? TRUE,
    ];

    $form['position'] = [
      '#type' => 'select',
      '#title' => $this->t('Toast position'),
      '#description' => $this->t('Choose where toasts should appear on the screen.'),
      '#options' => [
        'top-right' => $this->t('Top Right'),
        'top-left' => $this->t('Top Left'),
        'top-center' => $this->t('Top Center'),
        'bottom-right' => $this->t('Bottom Right'),
        'bottom-left' => $this->t('Bottom Left'),
        'bottom-center' => $this->t('Bottom Center'),
      ],
      '#default_value' => $config->get('position') ?? 'top-right',
    ];

    $form['type_mapping'] = [
      '#type' => 'details',
      '#title' => $this->t('Toast type to component mapping'),
      '#description' => $this->t('Configure toast types and their SDC components. Base types (status, warning, error) cannot be removed.'),
      '#open' => TRUE,
      '#tree' => TRUE,
      '#prefix' => '<div id="toast-type-mapping-wrapper">',
      '#suffix' => '</div>',
    ];

    // Get existing mappings.
    $type_mapping = $config->get('type_component_mapping') ?? [];

    // Base types that cannot be removed.
    $base_types = [
      'status' => [
        'component' => 'advanced_toast:toast-status',
        'label' => 'Status',
      ],
      'warning' => [
        'component' => 'advanced_toast:toast-warning',
        'label' => 'Warning',
      ],
      'error' => [
        'component' => 'advanced_toast:toast-error',
        'label' => 'Error',
      ],
    ];

    // Merge with existing custom mappings.
    $all_mappings = [];
    foreach ($base_types as $type => $defaults) {
      $all_mappings[$type] = [
        'type' => $type,
        'component' => $type_mapping[$type] ?? $defaults['component'],
        'is_base' => TRUE,
        'label' => $defaults['label'],
      ];
    }

    // Add custom types.
    foreach ($type_mapping as $type => $component) {
      if (!isset($base_types[$type])) {
        $all_mappings[$type] = [
          'type' => $type,
          'component' => $component,
          'is_base' => FALSE,
          'label' => NULL,
        ];
      }
    }

    // Build complete rows array (base + custom + empty).
    $rows = $form_state->get('toast_rows');
    if ($rows === NULL) {
      // Initialize with base types and custom types from config, plus 1 empty row.
      $rows = [];
      foreach ($all_mappings as $mapping) {
        $rows[] = $mapping;
      }
      // Add 1 empty row.
      $rows[] = [
        'type' => '',
        'component' => '',
        'is_base' => FALSE,
        'label' => NULL,
      ];
      $form_state->set('toast_rows', $rows);
    }

    $form['type_mapping']['mappings'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Toast Type'),
        $this->t('Component ID'),
        $this->t('Actions'),
      ],
      '#empty' => $this->t('No type mappings defined.'),
    ];

    // Render all rows from the array.
    foreach ($rows as $index => $row_data) {
      $form['type_mapping']['mappings'][$index] = $this->buildMappingRow(
        $index,
        $row_data['type'],
        $row_data['component'],
        $row_data['is_base'],
        $row_data['label']
      );
    }

    $form['type_mapping']['add_more'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add another row'),
      '#submit' => ['::addMoreRows'],
      '#ajax' => [
        'callback' => '::ajaxCallback',
        'wrapper' => 'toast-type-mapping-wrapper',
      ],
      '#limit_validation_errors' => [],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Build a single mapping row.
   *
   * @param int $row_index
   *   The row index in the table.
   * @param string $type
   *   The toast type.
   * @param string $component
   *   The component ID.
   * @param bool $is_base
   *   Whether this is a base type.
   * @param string|null $label
   *   The label for base types.
   *
   * @return array
   *   Form elements for the row.
   */
  protected function buildMappingRow($row_index, $type, $component, $is_base = FALSE, $label = NULL) {
    $row = [];
    $row['#attributes']['class'][] = $is_base ? 'toast-mapping-row--base' : 'toast-mapping-row--custom';

    // COLUMN 1: Toast Type.
    if ($is_base) {
      $row['toast_type'] = [
        '#type' => 'textfield',
        '#default_value' => $type,
        '#disabled' => TRUE,
        '#attributes' => ['class' => ['toast-type-disabled']],
      ];
    }
    else {
      $row['toast_type'] = [
        '#type' => 'textfield',
        '#default_value' => $type,
        '#placeholder' => $this->t('e.g., info'),
        '#attributes' => ['class' => ['toast-type-custom']],
        '#wrapper_attributes' => ['class' => ['toast-type-cell']],
      ];
    }

    // COLUMN 2: Component ID.
    $row['component'] = [
      '#type' => 'textfield',
      '#default_value' => $component,
      '#placeholder' => $this->t('e.g., mytheme:toast-info'),
      '#attributes' => ['class' => ['toast-component-id']],
    ];

    // COLUMN 3: Actions.
    if ($is_base) {
      // Base types show "Base type" label.
      $row['actions'] = [
        '#type' => 'markup',
        '#markup' => '<span class="toast-base-type-label">' . $this->t('Base type') . '</span>',
      ];
    }
    else {
      // Custom types always show Remove button (even empty rows).
      $row['actions'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove'),
        '#name' => 'remove_row_' . $row_index,
        '#submit' => ['::removeRow'],
        '#ajax' => [
          'callback' => '::ajaxCallback',
          'wrapper' => 'toast-type-mapping-wrapper',
        ],
        '#attributes' => [
          'class' => ['button', 'button--small', 'button--danger', 'toast-remove-btn'],
        ],
        '#limit_validation_errors' => [],
      ];
    }

    // Add hidden field for base type after the visible columns.
    // Hidden fields don't affect column rendering order.
    if ($is_base) {
      $row['type_hidden'] = [
        '#type' => 'hidden',
        '#value' => $type,
      ];
    }

    return $row;
  }

  /**
   * Submit handler for adding more rows.
   */
  public function addMoreRows(array &$form, FormStateInterface $form_state) {
    $rows = $form_state->get('toast_rows') ?? [];
    // Add a new empty row to the array.
    $rows[] = [
      'type' => '',
      'component' => '',
      'is_base' => FALSE,
      'label' => NULL,
    ];
    $form_state->set('toast_rows', $rows);
    $form_state->setRebuild();
  }

  /**
   * Submit handler for removing a row.
   */
  public function removeRow(array &$form, FormStateInterface $form_state) {
    $trigger = $form_state->getTriggeringElement();
    $row_index = (int) str_replace('remove_row_', '', $trigger['#name']);

    $rows = $form_state->get('toast_rows') ?? [];
    // Remove the row at the specified index.
    // Do NOT re-index - keep the original keys so button indices match.
    if (isset($rows[$row_index])) {
      unset($rows[$row_index]);
      $form_state->set('toast_rows', $rows);
    }

    $form_state->setRebuild();
  }

  /**
   * AJAX callback for add/remove operations.
   */
  public function ajaxCallback(array &$form, FormStateInterface $form_state) {
    return $form['type_mapping'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $mappings = $form_state->getValue(['type_mapping', 'mappings']) ?? [];

    foreach ($mappings as $index => $mapping) {
      // Get type from either visible field or hidden field.
      $type = trim($mapping['toast_type'] ?? $mapping['type_hidden'] ?? '');

      // Get component.
      $component = trim($mapping['component'] ?? '');

      // Skip empty rows.
      if (empty($type) && empty($component)) {
        continue;
      }

      // Both fields must be filled if one is.
      if (empty($type) && !empty($component)) {
        $form_state->setErrorByName("type_mapping][mappings][$index][toast_type",
          $this->t('Toast type is required when component is specified.'));
      }
      if (!empty($type) && empty($component)) {
        $form_state->setErrorByName("type_mapping][mappings][$index][component",
          $this->t('Component ID is required when toast type is specified.'));
      }

      // Validate component ID format.
      if (!empty($component) && strpos($component, ':') === FALSE) {
        $form_state->setErrorByName("type_mapping][mappings][$index][component",
          $this->t('Component ID must be in the format "namespace:component-name" (e.g., "mytheme:toast-info").'));
      }

      // Validate that the component exists and is a valid toast component.
      if (!empty($component) && strpos($component, ':') !== FALSE) {
        try {
          $component_definition = $this->componentPluginManager->find($component);

          // Get the raw plugin definition to access thirdPartySettings.
          $plugin_definition = $this->componentPluginManager->getDefinition($component);

          // Check if component is marked as a toast component via thirdPartySettings.
          $third_party = $plugin_definition['thirdPartySettings']['advanced_toast'] ?? NULL;
          $is_toast = $third_party['is_toast_component'] ?? FALSE;

          if (!$is_toast) {
            $form_state->setErrorByName("type_mapping][mappings][$index][component",
              $this->t('Component "@component" is not a toast component. Toast components must have "thirdPartySettings.advanced_toast.is_toast_component: true" in their component.yml file.', [
                '@component' => $component,
              ]));
          }

          // Validate that it's a toast component by checking metadata.
          $metadata = $component_definition->metadata ?? NULL;
          if ($metadata) {
            $props = $metadata->schema['properties'] ?? [];

            // A toast component must have at least a 'message' property.
            if (!isset($props['message'])) {
              $form_state->setErrorByName("type_mapping][mappings][$index][component",
                $this->t('Component "@component" is not a valid toast component. Toast components must have a "message" property defined in their schema.', [
                  '@component' => $component,
                ]));
            }
          }
        }
        catch (\Exception $e) {
          $form_state->setErrorByName("type_mapping][mappings][$index][component",
            $this->t('Component "@component" does not exist. Please check that the component is defined in a module or theme.', [
              '@component' => $component,
            ]));
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('advanced_toast.settings');

    $config->set('replace_drupal_messages', $form_state->getValue('replace_drupal_messages'));
    $config->set('default_duration', $form_state->getValue('default_duration'));
    $config->set('default_dismissible', $form_state->getValue('default_dismissible'));
    $config->set('position', $form_state->getValue('position'));

    // Build mappings from table.
    $type_component_mapping = [];
    $mappings = $form_state->getValue(['type_mapping', 'mappings']) ?? [];

    foreach ($mappings as $row) {
      // Get type from either visible field or hidden field.
      $type = trim($row['toast_type'] ?? $row['type_hidden'] ?? '');

      // Get component.
      $component = trim($row['component'] ?? '');

      // Skip empty rows.
      if (empty($type) || empty($component)) {
        continue;
      }

      // Add to component mapping.
      $type_component_mapping[$type] = $component;
    }

    $config->set('type_component_mapping', $type_component_mapping);
    $config->save();

    // Clear the rows array so it rebuilds from config on next load.
    $form_state->set('toast_rows', NULL);

    parent::submitForm($form, $form_state);
  }

}
