<?php

namespace Drupal\advanced_toast\TwigExtension;

use Drupal\advanced_toast\AdvancedToastService;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

/**
 * Twig extension for toast notifications.
 */
class AdvancedToastTwigExtension extends AbstractExtension {

  /**
   * The toast service.
   *
   * @var \Drupal\advanced_toast\AdvancedToastService
   */
  protected $toastService;

  /**
   * Constructs a AdvancedToastTwigExtension object.
   *
   * @param \Drupal\advanced_toast\AdvancedToastService $toast_service
   *   The toast service.
   */
  public function __construct(AdvancedToastService $toast_service) {
    $this->toastService = $toast_service;
  }

  /**
   * {@inheritdoc}
   */
  public function getFunctions() {
    return [
      new TwigFunction('toast', [$this, 'addToast']),
      new TwigFunction('toast_status', [$this, 'addStatusToast']),
      new TwigFunction('toast_warning', [$this, 'addWarningToast']),
      new TwigFunction('toast_error', [$this, 'addErrorToast']),
    ];
  }

  /**
   * Add a toast notification.
   *
   * @param string $message
   *   The message to display.
   * @param string $type
   *   The toast type.
   * @param array $options
   *   Additional options.
   *
   * @return string
   *   Empty string (function is called for side effects).
   */
  public function addToast(string $message, string $type = 'status', array $options = []): string {
    $this->toastService->addToast($message, $type, $options);
    return '';
  }

  /**
   * Add a status toast.
   *
   * @param string $message
   *   The message to display.
   * @param array $options
   *   Additional options.
   *
   * @return string
   *   Empty string.
   */
  public function addStatusToast(string $message, array $options = []): string {
    $this->toastService->status($message, $options);
    return '';
  }

  /**
   * Add a warning toast.
   *
   * @param string $message
   *   The message to display.
   * @param array $options
   *   Additional options.
   *
   * @return string
   *   Empty string.
   */
  public function addWarningToast(string $message, array $options = []): string {
    $this->toastService->warning($message, $options);
    return '';
  }

  /**
   * Add an error toast.
   *
   * @param string $message
   *   The message to display.
   * @param array $options
   *   Additional options.
   *
   * @return string
   *   Empty string.
   */
  public function addErrorToast(string $message, array $options = []): string {
    $this->toastService->error($message, $options);
    return '';
  }

}
