/**
 * @file
 * Conditionally hide admin UI features.
 */
(function (Drupal, once) {
  /**
   * Text to use with the toggle button.
   */
  const TOGGLE_TEXT = {
    show: Drupal.t('Show additional fields'),
    hide: Drupal.t('Hide additional fields'),
    description: Drupal.t(
      'Some form elements may be hidden. Switch this form between full and partial detail',
    ),
  };

  /**
   * Creates the toggle button element.
   *
   * @returns {HTMLButtonElement}
   *   The toggle button element.
   */
  const createToggle = () => {
    const toggleButton = document.createElement('button');
    toggleButton.className = 'advanced-form-toggle btn';
    toggleButton.type = 'button';
    toggleButton.title = TOGGLE_TEXT.description;
    toggleButton.textContent = TOGGLE_TEXT.show;
    return toggleButton;
  };

  /**
   * Wraps the toggle button in a container div.
   *
   * The purpose of the container is for styling.
   *
   * @param {HTMLButtonElement} toggle
   *   The toggle button element.
   *
   * @returns {HTMLDivElement}
   *   The wrapper element containing the toggle button.
   */
  const wrapToggle = (toggle) => {
    const wrapper = document.createElement('div');
    wrapper.className = 'advanced-form-wrapper';
    wrapper.prepend(toggle);
    return wrapper;
  };

  /**
   * Converts text to a CSS-safe class name.
   *
   * @param {string} text
   *   The text to convert.
   *
   * @returns {string}
   *   A class name safe for use in CSS.
   */
  const textToClassName = (text) => {
    return `selected-${text
      .toLowerCase()
      .replace(/%[0-9A-F]/gi, '')
      .replace(/ /g, '-')}`;
  };

  /**
   * Updates form classes based on currently selected options.
   *
   * @param {HTMLFormElement} form
   *   The form element.
   * @param {NodeList} selectWidgets
   *   Collection of select elements to check for selected options.
   * @param {Set} selectionClasses
   *   Set tracking the currently applied selection classes.
   */
  const updateFormClasses = (form, selectWidgets, selectionClasses) => {
    // Reset any previous selections.
    selectionClasses.forEach((className) => {
      form.classList.remove(className);
    });
    selectionClasses.clear();

    // Add classes for all currently selected options.
    selectWidgets.forEach((select) => {
      const selectedOptions = Array.from(select.options).filter(
        (option) => option.selected,
      );
      selectedOptions.forEach((option) => {
        const value = option.value;
        if (value && value !== '_none') {
          const className = textToClassName(option.textContent);
          form.classList.add(className);
          selectionClasses.add(className);
        }
      });
    });
  };

  /**
   * Drupal behavior for advanced form functionality.
   */
  Drupal.behaviors.advancedform = {
    attach(context) {
      const forms = once('advanced-form-toggle', '.node-form', context);
      forms.forEach((form) => {
        const selectionClasses = new Set();
        const selectWidgets = form.querySelectorAll(
          '.field--widget-options-select select',
        );

        const toggleButton = createToggle();
        const wrapper = wrapToggle(toggleButton);
        form.prepend(wrapper);

        toggleButton.addEventListener('click', () => {
          const isFiltered = form.classList.contains('advanced-form-filtered');
          form.classList.toggle('advanced-form-filtered');
          toggleButton.textContent = isFiltered
            ? TOGGLE_TEXT.hide
            : TOGGLE_TEXT.show;
        });

        updateFormClasses(form, selectWidgets, selectionClasses);

        selectWidgets.forEach((select) => {
          select.addEventListener('change', () => {
            updateFormClasses(form, selectWidgets, selectionClasses);
          });
        });
      });
    },
  };
})(Drupal, once);
