<?php

namespace Drupal\advancedqueue_sqs_backend\Plugin\AdvancedQueue\Backend;

use Aws\Credentials\CredentialProvider;
use Aws\Sqs\Exception\SqsException;
use Aws\Sqs\SqsClient;
use Drupal\advancedqueue_sqs_backend\SqsMessageMapper;
use Drupal\advancedqueue\Attribute\AdvancedQueueBackend;
use Drupal\advancedqueue\Job;
use Drupal\advancedqueue\Plugin\AdvancedQueue\Backend\BackendBase;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Provides the sqs queue backend.
 */
#[AdvancedQueueBackend(
  id: "sqs_backend",
  label: new TranslatableMarkup("SQS Backend"),
)]
class SqsBackend extends BackendBase {

  protected ?SqsClient $client = NULL;

  protected ?string $queueUrl = NULL;

  protected ?string $dlqQueueUrl = NULL;

  protected ?int $waitTimeSeconds = NULL;

  protected ?int $visibilityTimeout = NULL;

  /**
   * Get the AWS SQS client
   *
   * @return \Aws\Sqs\SqsClient|null
   */
  protected function getClient(): ?SqsClient {
    if ($this->client) {
      return $this->client;
    }

    $region = getenv('AWS_REGION');
    if ($this->configuration['region']) {
      $region = $this->configuration['region'];
    }

    // When running in a Lambda credential handling needs to be made explicit.
    $provider = CredentialProvider::defaultProvider();
    $config = [
      'region' => $region,
      'version' => 'latest',
      'credentials' => $provider,
    ];
    if ($this->configuration['aws_access_key'] && $this->configuration['aws_secret_access_key']) {
      $config['credentials'] = [
        'key' => $this->configuration['aws_access_key'],
        'secret' => $this->configuration['aws_secret_access_key'],
      ];
    }

    $this->client = new SqsClient($config);
    return $this->client;
  }

  /**
   * Get the URL for the AWS SQS queue.
   *
   * @return string|null
   */
  protected function getQueueUrl(): ?string {
    if (!$this->queueUrl) {
      $this->queueUrl = $this->configuration['sqs_queue_url'] ?? NULL;
    }
    return $this->queueUrl;
  }

  /**
   * Get the URL for the AWS SQS DLQ queue.
   *
   * @return string|null
   */
  protected function getDlqQueueUrl(): ?string {
    if (!$this->dlqQueueUrl) {
      $this->dlqQueueUrl = $this->configuration['sqs_dlq_queue_url'] ?? NULL;
    }
    return $this->dlqQueueUrl;
  }

  /**
   * Get the wait time seconds for the AWS SQS queue claim.
   *
   * @return int|null
   */
  protected function getWaitTimeSeconds(): ?int {
    if (!$this->waitTimeSeconds) {
      $this->waitTimeSeconds = $this->configuration['sqs_wait_time_seconds'] ?? 10;
    }
    return $this->waitTimeSeconds;
  }

  /**
   * Get the wait time seconds for the AWS SQS DLQ queue claim.
   *
   * @return int|null
   */
  protected function getVisibilityTimeout(): ?int {
    if (!$this->visibilityTimeout) {
      $this->visibilityTimeout = $this->configuration['sqs_visibility_timeout'] ?? 30;
    }
    return $this->visibilityTimeout;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['region'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Region'),
      '#description' => $this->t('The AWS Region, overrides (AWS_REGION).'),
      '#default_value' => $this->configuration['region'] ?? '',
      '#attributes' => ['autocomplete' => 'off'],
    ];

    $form['sqs_queue_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SQS Queue Url'),
      '#description' => $this->t('The Amazon Resource Name of your SQS queue.'),
      '#default_value' => $this->configuration['sqs_queue_url'] ?? '',
      '#required' => TRUE,
    ];

    $form['sqs_dlq_queue_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SQS DLQ Queue Url'),
      '#description' => $this->t('The Amazon Resource Name of your SQS queue\'s dead letter queue (optional).'),
      '#default_value' => $this->configuration['sqs_dlq_queue_url'] ?? '',
    ];

    $form['sqs_wait_time_seconds'] = [
      '#type' => 'number',
      '#title' => $this->t('Polling length in seconds (Wait time seconds)'),
      '#description' => $this->t('How long to allow a claim attempt to receive a message before determining there are no messages to claim.'),
      '#default_value' => $this->configuration['sqs_wait_time_seconds'] ?? 10,
      '#min' => 0,
      '#max' => 20, // AWS SQS limit
      '#step' => 1,
    ];

    $form['sqs_visibility_timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Item max processing time in seconds (Visibility timeout)'),
      '#description' => $this->t('The maximum amount of time allow the system to process a message before it is considered to have failed.'),
      '#default_value' => $this->configuration['sqs_visibility_timeout'] ?? 30,
      '#min' => 0,
      '#max' => 43200, // 12 hours, AWS SQS limit
      '#step' => 1,
    ];

    $form['aws_credentials'] = [
      '#type' => 'details',
      '#title' => $this->t('AWS Credentials'),
      '#open' => TRUE,
      '#description' => $this->t('If these are set in your container, environment variables, or ~/.aws/credentials, you do not need to set them here unless you wish to override the defaults.'),
    ];

    $form['aws_credentials']['aws_access_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Access Key'),
      '#description' => $this->t('The AWS credentials access key, overrides (AWS_ACCESS_KEY_ID).'),
      '#default_value' => $this->configuration['aws_access_key'] ?? '',
      '#attributes' => ['autocomplete' => 'off'],
    ];

    $form['aws_credentials']['aws_secret_access_key'] = [
      '#type' => 'password',
      '#title' => $this->t('Secret Access Key'),
      '#description' => $this->t('The AWS credentials secret access key, overrides (AWS_SECRET_ACCESS_KEY).'),
      '#default_value' => $this->configuration['aws_secret_access_key'] ?? '',
      '#attributes' => ['autocomplete' => 'off'],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);

      $this->configuration = [];
      $this->configuration['sqs_queue_url'] = $values['sqs_queue_url'];
      $this->configuration['sqs_dlq_queue_url'] = $values['sqs_dlq_queue_url'];
      $this->configuration['sqs_visibility_timeout'] = $values['sqs_visibility_timeout'];
      $this->configuration['sqs_wait_time_seconds'] = $values['sqs_wait_time_seconds'];
      $this->configuration['region'] = $values['region'];
      $this->configuration['aws_access_key'] = $values['aws_credentials']['aws_access_key'];
      $this->configuration['aws_secret_access_key'] = $values['aws_credentials']['aws_secret_access_key'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createQueue() {
    // Not Required.
  }

  /**
   * {@inheritdoc}
   */
  public function deleteQueue() {
    // Skipped as the queue is externally managed.
  }

  /**
   * {@inheritdoc}
   */
  public function cleanupQueue() {
    // Not Required.
  }

  /**
   * {@inheritdoc}
   */
  public function countJobs(): array {
    try {
      $result = $this->getClient()->getQueueAttributes([
        'QueueUrl' => $this->getQueueUrl(),
        'AttributeNames' => [
          'ApproximateNumberOfMessages',
          'ApproximateNumberOfMessagesNotVisible',
        ],
      ]);

      $queued = (int) $result->get('Attributes')['ApproximateNumberOfMessages'];
      $in_flight = (int) $result->get('Attributes')['ApproximateNumberOfMessagesNotVisible'];

      $status = [
        Job::STATE_QUEUED => $queued,
        Job::STATE_PROCESSING => $in_flight,
        // Job::STATE_SUCCESS => 'N/A' -> SQS does not track successful processing
      ];
    }
    catch (SqsException $e) {
      \Drupal::logger('advancedqueue_sqs_backend')->error(
        'SQS Queue size request failed (%error).',
        ['%error' => (string) $e]
      );

      $status = [
        Job::STATE_QUEUED => '?',
        Job::STATE_PROCESSING => '?',
      ];
    }

    // We use the DLQ to capture failed records.
    if ($this->getDlqQueueUrl()) {
      try {
        $result = $this->getClient()->getQueueAttributes([
          'QueueUrl' => $this->getDlqQueueUrl(),
          'AttributeNames' => ['ApproximateNumberOfMessages'],
        ]);

        $status[Job::STATE_FAILURE] = (int) $result->get('Attributes')['ApproximateNumberOfMessages'];
      }
      catch (SqsException $e) {
        \Drupal::logger('advancedqueue_sqs_backend')->error(
          'SQS DLQ size request failed (%error).',
          ['%error' => (string) $e]
        );

        $status[Job::STATE_FAILURE] = '?';
      }
    }

    return $status;
  }

  /**
   * {@inheritdoc}
   */
  public function enqueueJob(Job $job, $delay = 0): void {
    $job->setQueueId($this->queueId);
    $message = SqsMessageMapper::fromJob($job)->toMessageString();
    try {
      $this->getClient()->sendMessage([
        'QueueUrl' => $this->getQueueUrl(),
        'MessageBody' => $message,
      ]);
    }
    catch (SqsException $e) {
      \Drupal::logger('advancedqueue_sqs_backend')->error(
        'Job could not be enqueued: Message too large (%size KB).',
        ['%size' => round(strlen(Json::encode($message)) / 1024, 2)]
      );
      throw new \InvalidArgumentException('SQS message too large. Job not enqueued.');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function enqueueJobs(array $jobs, $delay = 0): void {
    // SQS supports batches of 10.
    $chunks = array_chunk($jobs, 10);
    foreach ($chunks as $chunk) {
      $batch = array_map(function($job, $i) {
        $job->setQueueId($this->queueId);
        return [
          'Id' => (string) $i,
          'MessageBody' => SqsMessageMapper::fromJob($job)->toMessageString(),
        ];
      }, $chunk, array_keys($chunk));

      // A batch to SQS cannot exceed 256k. If this chunk will be too large,
      // handle each message individually.
      $batch_size_too_large = strlen(json_encode($batch)) >= 256 * 1024;
      if ($batch_size_too_large) {
        foreach ($chunk as $job) {
          $this->enqueueJob($job);
        }
        continue;
      }

      $this->getClient()->sendMessageBatch([
        'QueueUrl' => $this->getQueueUrl(),
        'Entries' => $batch,
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function retryJob(Job $job, $delay = 0) {
    // SQS requeues messages and tracks their failures.
  }

  /**
   * {@inheritdoc}
   */
  public function claimJob(): ?Job {
    $response = $this->getClient()->receiveMessage([
      'QueueUrl' => $this->getQueueUrl(),
      'MaxNumberOfMessages' => 1,
      'WaitTimeSeconds' => $this->getWaitTimeSeconds(),
      'VisibilityTimeout' => $this->getVisibilityTimeout(),
    ]);

    $messages = $response->get('Messages');
    if (!$messages) {
      return NULL;
    }

    $message = reset($messages);
    return SqsMessageMapper::fromSqsMessageArray($message)->toJob();
  }

  /**
   * {@inheritdoc}
   */
  public function onSuccess(Job $job): void {
    // If this is being processed by a lambda + SQS the integration will handle
    // deletion automatically.
    if (getenv('_HANDLER')) {
      return;
    }

    $this->getClient()->deleteMessage([
      'QueueUrl' => $this->getQueueUrl(),
      'ReceiptHandle' => $job->getId(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function onFailure(Job $job): void {
    // If this is being processed by a lambda + SQS the integration will handle
    // requeuing automatically.
    if (getenv('_HANDLER')) {
      return;
    }

    try {
      // Changing VisibilityTimeout immediately returns the item to the queue.
      $this->getClient()->changeMessageVisibility([
        'QueueUrl' => $this->getQueueUrl(),
        'ReceiptHandle' => $job->getId(),
        'VisibilityTimeout' => 0,
      ]);
    }
    catch (Aws\Sqs\Exception\SqsException $e) {
      \Drupal::logger('advancedqueue_sqs_backend')->error(
        'Change message visibility failed (%error).',
        ['%error' => (string) $e]
      );
    }
  }

}
