<?php

namespace Drupal\advancedqueue_sqs_backend;

use Drupal\advancedqueue\Job;
use Drupal\Component\Serialization\Json;

class SqsMessageMapper {

  protected array $body = [];
  protected ?string $id = NULL;

  /**
   * Constructs a new SqsMessageMapper Object.
   *
   * @param array $body
   *   The body of the sqs message, including type, queueId and payload keys.
   * @param string|null $id
   *   The receipt handle/job id, used to delete or fail a message.
   */
  public function __construct(array $body, ?string $id = NULL) {
    $this->body = $body;
    $this->id = $id;
  }

  /**
   * Factory method to create an SqsMessageMapper object from a message array.
   *
   * @param array $message
   *   A message array received from AWS SQS and deserialized.
   *
   * @return \Drupal\advancedqueue_sqs_backend\SqsMessageMapper
   *   An SqsMessageMapper object representing the data from the received string.
   */
  public static function fromSqsMessageArray(array $message): SqsMessageMapper {
    // SQS message keys' case can vary based on how they are retrieved so we
    // lower them to normalize.
    $message_lower = array_change_key_case($message);

    $body = $message_lower['body'];
    if (gettype($body) == 'string') {
      $body = Json::decode($body);
    }

    $id = $message_lower['receipthandle'];
    return new self($body, $id);
  }

  /**
   * Convert this SqsMessageMapper object to an Advanced Queue Job.
   *
   * @return \Drupal\advancedqueue\Job
   */
  public function toJob(): Job {
    $definition = [
      'id' => $this->id,
      'state' => Job::STATE_QUEUED,
      'type' => $this->body['type'],
      'payload' => $this->body['payload'],
      'queue_id' => $this->body['queueId'],
    ];

    return new Job($definition);
  }

  /**
   * Factory method to create an SqsMessageMapper from an Advanced Queue Job.
   *
   * @param \Drupal\advancedqueue\Job $job
   *
   * @return \Drupal\advancedqueue_sqs_backend\SqsMessageMapper
   */
  public static function fromJob(Job $job): SqsMessageMapper {
    $body = [
      'type' => $job->getType(),
      'payload' => $job->getPayload(),
      'queueId' => $job->getQueueId(),
    ];
    return new self($body, $job->getId());
  }

  /**
   * Serialize this object as an SQS message body.
   *
   * @return string
   *   A JSON-encoded message body.
   */
  public function toMessageString(): string {
    return Json::encode($this->body);
  }

}
