<?php

use Drupal\Core\Command\BootableCommandTrait;
use Drupal\Core\DrupalKernel;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Routing\RouteObjectInterface;
use Symfony\Component\Routing\Route;
use React\EventLoop\Loop;
use React\ChildProcess\Process;
use Drupal\Core\Database\Database;

$autoloader = require __DIR__ . '/../../../../vendor/autoload.php';

class WorkerProcess {

  use BootableCommandTrait;

  private $queue;
  private $queueName;
  private $queueStorage;
  private $queueProcessor;
  private $kernel;

  public function __construct($queue_name) {
    $this->boot();
    $this->queueStorage = \Drupal::entityTypeManager()->getStorage('advancedqueue_queue');
    $this->queueProcessor = \Drupal::service('advancedqueue.processor');
    $this->queueName = $queue_name;
    $this->queue = $this->queueStorage->load($queue_name);
  }

  protected function boot() {
    $autoloader = require getenv('PROJECT_ROOT') . '/vendor/autoload.php';
    $site_path = getenv('SITE_PATH') ?: 'sites/default';
    chdir(getenv('DRUPAL_ROOT'));

    $request = Request::createFromGlobals();
    $request->attributes->set(RouteObjectInterface::ROUTE_OBJECT, new Route('<none>'));
    $request->attributes->set(RouteObjectInterface::ROUTE_NAME, '<none>');

    $this->kernel = DrupalKernel::createFromRequest($request, $autoloader, 'prod');
    $this->kernel::bootEnvironment();
    $this->kernel->setSitePath($site_path);
    \Drupal\Core\Site\Settings::initialize($this->kernel->getAppRoot(), $this->kernel->getSitePath(), $autoloader);

    $this->kernel->boot();
    $this->kernel->preHandle($request);

    // Workaround for missing plugin definitions
    $this->kernel->rebuildContainer();
  }

  public function runQueue(int &$failures, int $max_failures): void {
    if (!$this->queue) {
      echo "Queue '$this->queueName' not found.\n";
      $failures = $max_failures;
      return;
    }

    $job = $this->queue->getBackend()->claimJob();
    if (!$job) {
      $failures++;
      echo "No job found (count $failures/$max_failures)\n";
      sleep(5);
      return;
    }

    $failures = 0;

    $start = microtime(true);
    $this->queueProcessor->processJob($job, $this->queue);
    $end = microtime(true);

    $duration = round($end - $start, 3);
    echo "Processed job in {$duration}s\n";
  }

  public function loop(int $max_failures = 10): void {
    $loop = Loop::get();
    $failures = 0;

    $loop->addPeriodicTimer(0.1, function() use (&$loop, &$failures, $max_failures) {
      $this->runQueue($failures, $max_failures);
      if ($failures >= $max_failures) {
        echo "Queue empty. Shutting down.\n";
        $loop->stop();
      }
    });

    $loop->run();

    // Cleanup after loop finishes
    $this->shutdown();
  }

  protected function shutdown(): void {
    echo "Shutting down kernel and closing DB connection.\n";
    Database::closeConnection();
    if ($this->kernel) {
      $this->kernel->shutdown();
    }
  }

}

function spawnWorkers(string $queue_name, int $worker_count = 4): void {
  $loop = Loop::get();
  for ($i = 0; $i < $worker_count; $i++) {
    $cmd = "php " . __FILE__ . " -w " . escapeshellarg($queue_name);
    $process = new Process($cmd);
    $process->start($loop);

    $process->stdout->on('data', function ($chunk) use ($i) {
      echo "[Worker $i] $chunk";
    });

    $process->stderr->on('data', function ($chunk) use ($i) {
      fwrite(STDERR, "[Worker $i ERR] $chunk");
    });

    $process->on('exit', function ($exitCode) use ($i) {
      echo "[Worker $i] exited with code $exitCode\n";
    });
  }
  $loop->run();
}

// Parse command-line args
$options = getopt("w:");
if (isset($options['w'])) {
  // Worker mode
  $worker = new WorkerProcess($options['w']);
  $worker->loop();
}
else {
  // Parent process: spawn workers
  $queue_name = $argv[1] ?? 'default';
  $worker_count = (int)($argv[2] ?? 1);
  spawnWorkers($queue_name, $worker_count);
}
