<?php

namespace Drupal\aero_weather\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\aero_weather\Service\AeroWeatherApi;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\file\FileUsage\FileUsageInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;

/**
 * Provides an 'Aero Weather' block.
 *
 * @Block(
 *   id = "aero_weather_vertical_block",
 *   admin_label = @Translation("Aero Weather (Vertical)"),
 *   category = @Translation("Aero Weather")
 * )
 */
class AeroWeatherVerticalBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The exchange rate API service.
   *
   * @var \Drupal\aero_weather\Service\AeroWeatherApi
   */
  protected $apiService;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The file usage service.
   *
   * @var \Drupal\file\FileUsage\FileUsageInterface
   */
  protected $fileUsage;

  /**
   * The entity repository service.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * The cache service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The file URL generator service.
   *
   * Used to generate absolute or relative URLs for file entities.
   *
   * @var \Drupal\Core\Url\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * Constructs a new AeroWeatherVerticalBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\aero_weather\Service\AeroWeatherApi $api_service
   *   The exchange rate API service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\file\FileUsage\FileUsageInterface $file_usage
   *   The file usage service.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, AeroWeatherApi $api_service, EntityTypeManagerInterface $entity_type_manager, FileUsageInterface $file_usage, EntityRepositoryInterface $entity_repository, CacheBackendInterface $cache, DateFormatterInterface $date_formatter, FileUrlGeneratorInterface $file_url_generator) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->apiService = $api_service;
    $this->entityTypeManager = $entity_type_manager;
    $this->fileUsage = $file_usage;
    $this->entityRepository = $entity_repository;
    $this->cache = $cache;
    $this->dateFormatter = $date_formatter;
    $this->fileUrlGenerator = $file_url_generator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('aero_weather.api'),
      $container->get('entity_type.manager'),
      $container->get('file.usage'),
      $container->get('entity.repository'),
      $container->get('cache.default'),
      $container->get('date.formatter'),
      $container->get('file_url_generator')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Get selected countries with their order.
    $location_name = $this->configuration['location_name'];
    $layout = $this->configuration['layout'];
    $date_format = $this->configuration['date_format'];
    $color_palette = $this->configuration['color_palette'];
    $show_forecast = $this->configuration['show_forecast'];
    $round_border = $this->configuration['round_border'];
    $forecast_days = $this->configuration['forecast_days'];

    $weather_data = $this->apiService->getWeatherData($location_name, $forecast_days);
    $icon_settings = $this->apiService->getWeatherIconSettings();

    // Check if the timestamp exists before attempting to format it.
    $formatted_date = '';
    if (isset($weather_data['location']['localtime_epoch'])) {
      // Use the injected date formatter service to convert the epoch timestamp
      // into a human-readable string based on the selected format.
      $formatted_date = $this->dateFormatter->format($weather_data['location']['localtime_epoch'], $date_format);
    }

    $uuid = aero_weather_generate_weather_card_uuid();

    // Get uploaded background_image URL if exists.
    $background_image_url = '';
    if (!empty($this->configuration['background_image'])) {
      $file = $this->entityTypeManager->getStorage('file')->load($this->configuration['background_image']);
      if ($file) {
        $background_image_url = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
      }
    }

    return [
      '#theme' => 'aero_weather_vertical_block',
      '#weather_data' => $weather_data,
      '#config' => [
        'layout' => $layout,
        'formatted_date' => $formatted_date,
        'color_palette' => $color_palette,
        'show_forecast' => $show_forecast,
        'round_border' => $round_border,
        'forecast_days' => $forecast_days,
        'uuid' => $uuid,
        'background_image_url' => $background_image_url,
        'icon_settings' => $icon_settings,
      ],
      '#cache' => [
        'keys' => ['aero_weather', 'block_data'],
        'tags' => ['config:aero_weather.settings'],
        'max-age' => 0,
      ],
      '#attached' => [
        'library' => [
          'aero_weather/color_palettes',
          'aero_weather/aero_weather_vertical',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form['settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Weather Widget Settings'),
      '#description' => $this->t('Configure the display options for the weather widget.'),
    ];

    $form['settings']['location_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Location Name'),
      '#description' => $this->t('Enter the valid name of the location (e.g., New York, USA) for which weather data will be displayed.'),
      '#default_value' => $this->configuration['location_name'] ?? '',
      '#required' => TRUE,
    ];

    $form['settings']['layout'] = [
      '#type' => 'select',
      '#title' => $this->t('Layout'),
      '#description' => $this->t('Select the layout style for the weather card display.'),
      '#default_value' => $this->configuration['layout'] ?? '',
      '#options' => [
        'layout-1' => $this->t('Layout 1'),
        'layout-2' => $this->t('Layout 2'),
        'layout-3' => $this->t('Layout 3'),
      ],
    ];

    // Load available date formats.
    $date_format_storage = $this->entityTypeManager->getStorage('date_format');
    $formats = $date_format_storage->loadMultiple();

    $date_format_options = [];
    foreach ($formats as $format) {
      $date_format_options[$format->id()] = $format->label();
    }

    $form['settings']['date_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Date format'),
      '#description' => $this->t('Select how dates should be displayed throughout this weather block.'),
      '#options' => $date_format_options,
      '#default_value' => $this->configuration['date_format'] ?? '',
    ];

    $form['settings']['color_palette'] = [
      '#type' => 'select',
      '#title' => $this->t('Color Palette'),
      '#description' => $this->t('Choose a predefined color palette to customize the appearance of the widget.'),
      '#options' => aero_weather_color_palettes(),
      '#default_value' => $this->configuration['color_palette'] ?? 'palette-1',
    ];

    $form['settings']['show_forecast'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Forecast'),
      '#description' => $this->t('Enable to display a multi-day weather forecast.'),
      '#default_value' => $this->configuration['show_forecast'] ?? 0,
    ];

    $form['settings']['forecast_days'] = [
      '#type' => 'select',
      '#title' => $this->t('Forecast Days'),
      '#description' => $this->t('Select the number of days to display in the forecast. Only shown if "Show Forecast" is enabled.'),
      '#options' => [
        3 => $this->t('3 days'),
        5 => $this->t('5 days'),
        10 => $this->t('10 days'),
        15 => $this->t('15 days'),
      ],
      '#default_value' => $this->configuration['forecast_days'] ?? 3,
      '#states' => [
        'visible' => [
          ':input[name="settings[settings][show_forecast]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['settings']['round_border'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Round Border'),
      '#description' => $this->t('Enable to apply rounded corners to the weather card.'),
      '#default_value' => $this->configuration['round_border'] ?? 0,
    ];

    $form['settings']['background_image'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Upload Image'),
      '#description' => $this->t('Allowed file types: jpg, jpeg, png, gif, webp.'),
      '#default_value' => isset($this->configuration['background_image']) ? [$this->configuration['background_image']] : NULL,
      '#upload_location' => 'public://aero_weather_images/',
      '#upload_validators'  => [
        'FileExtension' => ['extensions' => 'gif png jpg jpeg svg webp'],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    $values = $form_state->getValue('settings');

    // Handle background_image upload.
    if (!empty($values['background_image'][0])) {
      $fid = $values['background_image'][0];
      $file = $this->entityTypeManager->getStorage('file')->load($fid);
      if ($file) {
        $file->setPermanent();
        $file->save();
      }

      // If replacing an old background_image, set old file to temporary.
      if (!empty($this->configuration['background_image']) && $this->configuration['background_image'] != $fid) {
        $old_file = $this->entityTypeManager->getStorage('file')->load($this->configuration['background_image']);
        if ($old_file) {
          $old_file->setTemporary();
          $old_file->save();
        }
      }

      $this->configuration['background_image'] = $fid;
    }
    else {
      // If background_image is removed, set old file to temporary.
      if (!empty($this->configuration['background_image'])) {
        $old_file = $this->entityTypeManager->getStorage('file')->load($this->configuration['background_image']);
        if ($old_file) {
          $old_file->setTemporary();
          $old_file->save();
        }
      }
      $this->configuration['background_image'] = NULL;
    }

    $this->configuration['location_name'] = $values['location_name'];
    $this->configuration['layout'] = $values['layout'];
    $this->configuration['color_palette'] = $values['color_palette'];
    $this->configuration['show_forecast'] = $values['show_forecast'];
    $this->configuration['date_format'] = $values['date_format'];
    $this->configuration['round_border'] = $values['round_border'];
    $this->configuration['forecast_days'] = $values['forecast_days'];
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'layout' => 'layout-1',
      'date_format' => '',
      'color_palette' => 'palette-1',
      'show_forecast' => TRUE,
      'round_border' => TRUE,
      'forecast_days' => '5',
      'background_image' => NULL,
    ] + parent::defaultConfiguration();
  }

}
