<?php

namespace Drupal\affiliated\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Cookie;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Class AffiliateUserPagesController.
 */
class AffiliatedTrackerController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * @var \Drupal\affiliated\AffiliateManager
   */
  protected $affiliateManager;

  /**
   * The affiliate config settings.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $config;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->currentUser = $container->get('current_user');
    $instance->affiliateManager = $container->get('affiliate.manager');
    $instance->config = $container->get('config.factory')->get('affiliated.settings');
    return $instance;
  }

  /**
   * Track visits from affiliate links.
   */
  public function track(Request $request): JsonResponse {
    $params = $request->request->all();
    $response = new JsonResponse();
    $responseData['tracked'] = FALSE;

    if (!$this->shouldTrackUser()) {
      $responseData['message'] = 'Tracking disabled for user';
      $response->setData($responseData);
      return $response;
    }

    $affiliate_code = $params['affiliate'] ?? NULL;
    if ($affiliate_code) {
      $stored_affiliate_code = $this->affiliateManager->getStoredAffiliateCode();
      if (!$stored_affiliate_code || ($this->config->get('click_precedence') == 'overwrite' && $affiliate_code != $stored_affiliate_code)) {
        /** @var \Drupal\user\UserInterface $affiliate */
        $affiliate = $this->affiliateManager->getAccountFromCode($affiliate_code);
        $campaign_code = $params['campaign'] ?? NULL;
        if ($affiliate) {
          $campaign = $this->affiliateManager->getCampaignFromCode($campaign_code);
          $click = $this->affiliateManager->registerClick($affiliate, $campaign, $params['landingPage'], $params['referrerUrl']);
          // If our click was successful, set a cookie.
          if ($click) {
            $responseData['affiliate_id'] = $affiliate_code;
            $responseData['affiliate_campaign'] = $campaign->id();
            $responseData['tracked'] = TRUE;
            $cookie_lifetime = strtotime('+' . $this->config->get('cookie_lifetime'));
            $response->headers->setCookie(Cookie::create('affiliate_id', $affiliate_code, $cookie_lifetime));
            $response->headers->setCookie(Cookie::create('affiliate_campaign', $campaign->id(), $cookie_lifetime));
          }
        }
      }
    }
    $response->setData($responseData);
    return $response;
  }

  /**
   * Check if the current user should be tracked.
   *
   * @return bool
   *   TRUE if user should be tracked, FALSE otherwise.
   */
  protected function shouldTrackUser(): bool {
    // Check if user has excluded roles.
    $exclude_roles = $this->config->get('exclude_roles') ?? [];
    if (!empty($exclude_roles)) {
      $user_roles = $this->currentUser->getRoles();
      if (array_intersect($user_roles, $exclude_roles)) {
        return FALSE;
      }
    }
    return TRUE;
  }

}
