<?php

namespace Drupal\affiliated\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class AffiliateSettingsForm.
 *
 * @ingroup affiliate
 */
class AffiliateSettingsForm extends ConfigFormBase {

  /**
   * The Entity Type Manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  public function __construct(
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    parent::__construct($config_factory);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      'affiliated.settings',
    ];
  }

  /**
   * Returns a unique string identifying the form.
   *
   * @return string
   *   The unique string identifying the form.
   */
  public function getFormId(): string {
    return 'affiliated_settings';
  }

  /**
   * Form submission handler.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Empty implementation of the abstract submit class.
    $this->config('affiliated.settings')
      ->set('cookie_lifetime', $form_state->getValue('cookie_lifetime'))
      ->set('affiliate_key', $form_state->getValue('affiliate_key'))
      ->set('campaign_key', $form_state->getValue('campaign_key'))
      ->set('allow_owner', $form_state->getValue('allow_owner'))
      ->set('click_precedence', $form_state->getValue('click_precedence'))
      ->set('affiliate_code_type', $form_state->getValue('affiliate_code_type'))
      ->set('overview', $form_state->getValue('overview'))
      ->set('path_visibility_paths', $form_state->getValue('path_visibility_paths'))
      ->set('path_visibility_mode', $form_state->getValue('path_visibility_mode'))
      ->set('exclude_roles', array_filter($form_state->getValue('exclude_roles')))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Defines the settings form for affiliates.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   Form definition array.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('affiliated.settings');
    $form['tabs'] = [
      '#type' => 'vertical_tabs',
      '#tree' => FALSE,
    ];
    $form['basic'] = [
      '#type' => 'details',
      '#group' => 'tabs',
      '#tree' => FALSE,
      '#title' => $this->t('Basic settings'),
    ];
    $form['basic']['cookie_lifetime'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Cookie Lifetime'),
      '#description' => $this->t('How long should the cookie be valid. <a href="http://php.net/strtotime">Strtotime</a> syntax. The value 0 means "until the browser is closed."'),
      '#default_value' => $config->get('cookie_lifetime'),
      '#required' => TRUE,
      '#group' => 'basic',
    ];
    $form['basic']['affiliate_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Affiliate ID key'),
      '#description' => $this->t('The url variable for the affiliate id'),
      '#default_value' => $config->get('affiliate_key'),
      '#required' => TRUE,
      '#group' => 'basic',
    ];
    $form['basic']['affiliate_code_type'] = [
      '#type' => 'radios',
      '#title' => $this->t('Affiliate variable type'),
      '#options' => [
        'user_id' => $this->t('User ID'),
        'username' => $this->t('Username'),
      ],
      '#description' => $this->t('The type of variable to use as the affiliate id.'),
      '#default_value' => $config->get('affiliate_code_type'),
      '#required' => TRUE,
      '#group' => 'basic',
    ];
    $form['basic']['campaign_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Campaign ID key'),
      '#description' => $this->t('The url variable for the campaign id'),
      '#default_value' => $config->get('campaign_key'),
      '#required' => TRUE,
      '#group' => 'basic',
    ];
    $form['basic']['allow_owner'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Record clicks and conversions from own account.'),
      '#description' => $this->t('Allow affiliates to act on their own links.'),
      '#default_value' => $config->get('allow_owner'),
      '#group' => 'basic',
    ];
    $form['basic']['click_precedence'] = [
      '#type' => 'radios',
      '#title' => $this->t('Click precedence'),
      '#options' => [
        'overwrite' => $this->t('Overwrite the affiliate cookie (new visit takes precedence)'),
        'deny' => $this->t('Reject the affiliate cookie (first visit takes precedence)'),
      ],
      '#description' => $this->t('When a user that already has an affiliate cookie visits the site through an affiliate link.'),
      '#default_value' => $config->get('click_precedence'),
      '#required' => TRUE,
      '#group' => 'basic',
    ];
    $form['affiliate_center'] = [
      '#type' => 'details',
      '#group' => 'tabs',
      '#tree' => FALSE,
      '#title' => $this->t('Affiliate Center'),
    ];
    $overview_active = $config->get('overview');
    $form['affiliate_center']['overview'] = [
      '#base_type' => 'textarea',
      '#type' => 'text_format',
      '#format' => $overview_active['format'] ?? 'basic_html',
      '#title' => $this->t('Affiliate Center Overview'),
      '#default_value' => $overview_active['value'] ?? '',
      '#description' => $this->t('Text shown on the affiliate dashboard. The token <strong>!affiliate_params</strong> will print the affiliate url parameters.'),
    ];
    $form['visibility'] = [
      '#type' => 'details',
      '#group' => 'tabs',
      '#tree' => FALSE,
      '#title' => $this->t('Visibility'),
    ];
    // Page specific visibility configurations.
    $visibility_request_path_pages = $config->get('path_visibility_paths');

    $description = $this->t("Specify pages by using their paths. Enter one path per line. The '*' character is a wildcard. Example paths are %blog for the blog page and %blog-wildcard for every personal blog. %front is the front page.", [
      '%blog' => '/blog',
      '%blog-wildcard' => '/blog/*',
      '%front' => '<front>',
    ]);

    $form['visibility']['path_visibility_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Add tracking to specific pages'),
      '#options' => [
        'all_pages' => $this->t('Every page except the listed pages'),
        'listed_pages' => $this->t('The listed pages only'),
      ],
      '#default_value' => $config->get('path_visibility_mode'),
      '#group' => 'visibility',
    ];
    $form['visibility']['path_visibility_paths'] = [
      '#type' => 'textarea',
      '#title' => 'Pages',
      '#title_display' => 'invisible',
      '#default_value' => !empty($visibility_request_path_pages) ? $visibility_request_path_pages : '',
      '#description' => $description,
      '#rows' => 10,
      '#group' => 'visibility',
    ];

    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    $role_options = [];
    foreach ($roles as $role) {
      $role_options[$role->id()] = $role->label();
    }

    $form['visibility']['exclude_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Exclude roles from tracking'),
      '#description' => $this->t('Users with the selected roles will not be tracked.'),
      '#options' => $role_options,
      '#default_value' => $config->get('exclude_roles') ?? [],
      '#group' => 'visibility',
    ];

    return parent::buildForm($form, $form_state);
  }

}
