<?php

namespace Drupal\affiliated\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for affiliate user pages and dashboard.
 */
class AffiliateUserPagesController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The affiliate manager service.
   *
   * @var \Drupal\affiliated\AffiliateManager
   */
  protected $affiliateManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->currentUser = $container->get('current_user');
    $instance->affiliateManager = $container->get('affiliate.manager');
    return $instance;
  }

  /**
   * Affiliate Campaign add form page callback.
   */
  public function addCampaignPage(UserInterface $user): array {
    // Show a create campaign entity form.
    $campaign = $this->entityTypeManager()->getStorage('affiliate_campaign')
      ->create([
        'affiliate' => $user->id(),
      ]);

    // Get the entity form.
    $formObject = $this->entityTypeManager()
      ->getFormObject('affiliate_campaign', 'add')
      ->setEntity($campaign);

    $build['form'] = $this->formBuilder()->getForm($formObject);

    // Hide the default and user_id fields, they are only relevant on the
    // global campaign form.
    $build['form']['user_id']['#access'] = FALSE;
    $build['form']['is_default']['#access'] = FALSE;

    return $build;
  }

  /**
   * Affiliate campaign overview page access.
   */
  public function addCampaignPageAccess(?UserInterface $user = NULL) {
    if ($user->id() == $this->currentUser->id() || ($this->currentUser->hasPermission('administer users') || $this->currentUser->hasPermission('manage affiliates'))) {
      return AccessResult::allowedIfHasPermissions($user, [
        'create affiliate_campaign entities',
        'act as an affiliate',
      ], 'AND');
    }
  }

  /**
   * Affiliate Center page callback.
   */
  public function overviewPage(UserInterface $user): array {
    $build['overview'] = [
      '#theme' => 'affiliated_overview',
      '#affiliate' => $user,
    ];

    return $build;
  }

  /**
   * Access callback for overview page.
   */
  public function overviewPageAccess(UserInterface $user) {
    if ($this->currentUser->id() == $user->id()) {
      return AccessResult::allowedIfHasPermission($user, 'access affiliate center');
    }
    elseif ($this->currentUser->hasPermission('administer users') || $this->currentUser->hasPermission('manage affiliates')) {
      return AccessResult::allowed()->cachePerPermissions();
    }

    return AccessResult::forbidden()->cachePerPermissions();
  }

}
