<?php

namespace Drupal\affiliated\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Cookie;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for tracking affiliate link visits.
 */
class AffiliatedTrackerController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The affiliate manager service.
   *
   * @var \Drupal\affiliated\AffiliateManager
   */
  protected $affiliateManager;

  /**
   * The affiliate config settings.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $config;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->currentUser = $container->get('current_user');
    $instance->affiliateManager = $container->get('affiliate.manager');
    $instance->config = $container->get('config.factory')->get('affiliated.settings');
    return $instance;
  }

  /**
   * Track visits from affiliate links.
   */
  public function track(Request $request): JsonResponse {
    $response = new JsonResponse();
    $responseData['tracked'] = FALSE;

    if (!$this->shouldTrackUser()) {
      $responseData['message'] = 'Tracking disabled for user';
      $response->setStatusCode(400);
      $response->setData($responseData);
      return $response;
    }

    $params = $request->request->all();

    // Basic check that affiliate exists.
    $affiliate_code = $params['affiliate'] ?? NULL;
    if (empty($affiliate_code)) {
      $responseData['message'] = 'Missing affiliate code';
      $response->setStatusCode(400);
      $response->setData($responseData);
      return $response;
    }

    $stored_affiliate_code = $this->affiliateManager->getStoredAffiliateCode();
    $stored_campaign_code = $this->affiliateManager->getStoredCampaignCode();
    $campaign_code = $params['campaign'] ?? NULL;

    $should_track = FALSE;

    // No existing tracking - first visit.
    if (!$stored_affiliate_code) {
      $should_track = TRUE;
    }
    // Overwrite mode - update if affiliate OR campaign changed.
    elseif ($this->config->get('click_precedence') == 'overwrite') {
      if ($affiliate_code != $stored_affiliate_code || $campaign_code != $stored_campaign_code) {
        $should_track = TRUE;
      }
    }

    if ($should_track) {
      $affiliate = $this->affiliateManager->getAccountFromCode($affiliate_code);

      if ($affiliate) {
        // Ensure landing page and referrer are strings.
        $landing_page_param = $params['landingPage'] ?? '';
        $landing_page = is_string($landing_page_param) ? $landing_page_param : '';
        $referrer_param = $params['referrerUrl'] ?? '';
        $referrer = is_string($referrer_param) ? $referrer_param : '';

        $campaign = $this->affiliateManager->getCampaignFromCode($campaign_code);

        // Non-global campaigns can only be used by their owner.
        if ($campaign && !$campaign->isGlobal() && $campaign->getOwnerId() != $affiliate->id()) {
          $campaign = $this->affiliateManager->getDefaultCampaign();
        }

        // Campaign is required - if none exists, we can't track.
        if (!$campaign) {
          $responseData['message'] = 'No campaign available';
          $response->setData($responseData);
          return $response;
        }

        $click = $this->affiliateManager->registerClick(
          $affiliate,
          $campaign,
          $landing_page,
          $referrer
        );

        if ($click) {
          $responseData['affiliate_id'] = $affiliate_code;
          $responseData['affiliate_campaign'] = $campaign->id();
          $responseData['tracked'] = TRUE;

          $cookie_lifetime = strtotime('+' . $this->config->get('cookie_lifetime'));
          $response->headers->setCookie(
            Cookie::create('affiliate_id', $affiliate_code, $cookie_lifetime)
          );
          $response->headers->setCookie(
            Cookie::create('affiliate_campaign', $campaign->id(), $cookie_lifetime)
          );
        }
      }
    }

    $response->setData($responseData);
    return $response;
  }

  /**
   * Check if the current user should be tracked.
   *
   * @return bool
   *   TRUE if user should be tracked, FALSE otherwise.
   */
  protected function shouldTrackUser(): bool {
    // Check if user has excluded roles.
    $exclude_roles = $this->config->get('exclude_roles') ?? [];
    if (!empty($exclude_roles)) {
      $user_roles = $this->currentUser->getRoles();
      if (array_intersect($user_roles, $exclude_roles)) {
        return FALSE;
      }
    }
    return TRUE;
  }

}
