<?php

namespace Drupal\affiliated\Entity;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\user\EntityOwnerInterface;

/**
 * Interface for Affiliate Conversion entities.
 */
interface AffiliateConversionInterface extends ContentEntityInterface, EntityChangedInterface, EntityOwnerInterface, EntityPublishedInterface {

  /**
   * Gets the bundle entity for this conversion.
   *
   * @return \Drupal\affiliated\Entity\AffiliateConversionTypeInterface
   *   The conversion type bundle entity.
   */
  public function getBundle();

  /**
   * Gets the affiliate user entity.
   *
   * @return \Drupal\user\UserInterface|\Drupal\Core\Session\AnonymousUserSession
   *   The affiliate user entity, or anonymous session if no affiliate is set.
   */
  public function getAffiliate();

  /**
   * Gets the affiliate user ID.
   *
   * @return int|null
   *   The affiliate user ID, or NULL if not set.
   */
  public function getAffiliateId();

  /**
   * Gets the campaign entity.
   *
   * @return \Drupal\affiliated\Entity\AffiliateCampaignInterface|null
   *   The campaign entity, or NULL if not set.
   */
  public function getCampaign();

  /**
   * Gets the campaign entity ID.
   *
   * @return int|null
   *   The campaign entity ID, or NULL if not set.
   */
  public function getCampaignId();

  /**
   * Gets the parent entity associated with this conversion.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The parent entity, or NULL if not set or cannot be loaded.
   */
  public function getParentEntity();

  /**
   * Gets the parent entity type ID.
   *
   * @return string|null
   *   The parent entity type ID, or NULL if not set.
   */
  public function getParentEntityTypeId();

  /**
   * Gets the parent entity ID.
   *
   * @return string|null
   *   The parent entity ID, or NULL if not set.
   */
  public function getParentEntityId();

  /**
   * Sets the parent entity for this conversion.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The parent entity to associate with this conversion.
   */
  public function setParentEntity(EntityInterface $entity);

  /**
   * Sets the commission for a conversion.
   *
   * @param float $value
   *   The numeric value for the commission.
   * @param string $currency
   *   The currency of the value.
   */
  public function setCommission(float $value, string $currency = '');

  /**
   * Gets the commission amount and currency.
   *
   * @return array
   *   An associative array with:
   *   - amount: The commission amount (float).
   *   - currency: The currency code (string).
   */
  public function getCommission();

  /**
   * Checks if the conversion is approved.
   *
   * @return bool
   *   TRUE if the conversion is published/approved, FALSE otherwise.
   */
  public function isApproved();

  /**
   * Gets the admin notes.
   *
   * @return string|null
   *   The admin notes, or NULL if not set.
   */
  public function getNotes(): ?string;

  /**
   * Sets the admin notes.
   *
   * @param string|null $notes
   *   The notes, or NULL to clear.
   *
   * @return static
   */
  public function setNotes(?string $notes): static;

  /**
   * Cancels the conversion with a reason.
   *
   * This is a convenience method that unpublishes the conversion and sets
   * a note explaining why.
   *
   * @param string $reason
   *   The cancellation reason (e.g., 'refund', 'fraud', 'duplicate').
   *
   * @return static
   */
  public function cancel(string $reason): static;

  /**
   * Gets the rejection reason if the conversion was rejected during save.
   *
   * This method returns the reason if save() returned FALSE due to a
   * ConversionPreCreateEvent subscriber rejecting the conversion.
   *
   * @return string|null
   *   The rejection reason, or NULL if not rejected.
   */
  public function getRejectionReason(): ?string;

}
