<?php

namespace Drupal\affiliated\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBundleBase;

/**
 * Defines the affiliate Conversion type entity.
 *
 * @ConfigEntityType(
 *   id = "affiliate_conversion_type",
 *   label = @Translation("Affiliate Conversion type"),
 *   handlers = {
 *     "access" = "Drupal\affiliated\Entity\Handler\AffiliateConversionTypeAccessControlHandler",
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" =
 *   "Drupal\affiliated\Entity\Handler\AffiliateConversionTypeListBuilder",
 *     "form" = {
 *       "default" = "Drupal\affiliated\Form\AffiliateConversionTypeForm",
 *       "add" = "Drupal\affiliated\Form\AffiliateConversionTypeForm",
 *       "edit" = "Drupal\affiliated\Form\AffiliateConversionTypeForm",
 *       "delete" = "Drupal\affiliated\Form\AffiliateConversionTypeDeleteForm"
 *     },
 *     "route_provider" = {
 *        "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *      },
 *   },
 *   config_prefix = "affiliate_conversion_type",
 *   admin_permission = "administer affiliate_conversion types",
 *   bundle_of = "affiliate_conversion",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "weight" = "weight",
 *     "uuid" = "uuid"
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "uuid",
 *     "description",
 *     "default_commission",
 *     "default_status",
 *     "label_pattern",
 *   },
 *   links = {
 *     "canonical" = "/admin/structure/affiliate/conversion/types/{affiliate_conversion_type}",
 *     "add-form" = "/admin/structure/affiliate/conversion/types/add",
 *     "edit-form" = "/admin/structure/affiliate/conversion/types/{affiliate_conversion_type}/edit",
 *     "delete-form" = "/admin/structure/affiliate/conversion/types/{affiliate_conversion_type}/delete",
 *     "collection" = "/admin/structure/affiliate/conversion/types"
 *   }
 * )
 */
class AffiliateConversionType extends ConfigEntityBundleBase implements AffiliateConversionTypeInterface {

  /**
   * The affiliate Conversion type ID.
   *
   * @var string
   */
  protected $id;

  /**
   * The affiliate Conversion type label.
   *
   * @var string
   */
  protected $label;

  /**
   * The affiliate Conversion type description.
   *
   * @var string
   */
  protected $description;

  /**
   * The conversion label token pattern.
   *
   * @var string
   */
  protected $label_pattern;

  /**
   * The default commission amount.
   *
   * @var float|null
   */
  protected $default_commission;

  /**
   * Whether new conversions are published by default.
   *
   * @var bool
   */
  protected $default_status = TRUE;

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->description;
  }

  /**
   * Returns the string representation of this entity type.
   *
   * @return string
   *   The label of the entity type.
   */
  public function __toString() {
    return (string) $this->label();
  }

  /**
   * Checks whether this is the default revision.
   *
   * Config entities are not revisionable, so this always returns TRUE.
   * Required by EntityViewBuilder.
   *
   * @return bool
   *   Always TRUE for config entities.
   */
  public function isDefaultRevision(): true {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function setDescription($description) {
    $this->set('description', $description);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getLabelPattern() {
    return (string) $this->label_pattern ?? '';
  }

  /**
   * Gets the default commission amount.
   *
   * @return float|null
   *   The default commission amount, or NULL if not set.
   */
  public function getDefaultCommission(): ?float {
    return $this->default_commission;
  }

  /**
   * Gets whether new conversions should be published by default.
   *
   * @return bool
   *   TRUE if new conversions are published by default, FALSE otherwise.
   */
  public function getDefaultStatus(): bool {
    return (bool) $this->default_status;
  }

}
