<?php

namespace Drupal\affiliated\Entity\Handler;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Access controller for the AffiliateCampaign entity.
 */
class AffiliateCampaignAccessControlHandler extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected $viewLabelOperation = TRUE;

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    /** @var \Drupal\affiliated\Entity\AffiliateCampaignInterface $entity */

    switch ($operation) {
      case 'view label':
        return AccessResult::allowed()
          ->addCacheableDependency($entity);

      case 'view':
        if ($account->hasPermission('view any affiliate_campaign entities')) {
          return AccessResult::allowed()
            ->cachePerPermissions()
            ->addCacheableDependency($entity);
        }

        if ($this->checkOwn($entity, $operation, $account)) {
          return AccessResult::allowed()
            ->cachePerPermissions()
            ->cachePerUser()
            ->addCacheableDependency($entity);
        }

        return AccessResult::neutral()
          ->cachePerPermissions()
          ->addCacheableDependency($entity);

      case 'update':
        if ($account->hasPermission('edit any affiliate_campaign entities')) {
          return AccessResult::allowed()
            ->cachePerPermissions()
            ->addCacheableDependency($entity);
        }

        // Owners can always edit their own campaigns (even if global).
        if ($this->checkOwn($entity, $operation, $account)) {
          return AccessResult::allowed()
            ->cachePerPermissions()
            ->cachePerUser()
            ->addCacheableDependency($entity);
        }

        return AccessResult::neutral()
          ->cachePerPermissions()
          ->addCacheableDependency($entity);

      case 'delete':
        // The default campaign cannot be deleted.
        if ($entity->isDefault()) {
          return AccessResult::forbidden('Cannot delete default campaign')
            ->addCacheableDependency($entity);
        }

        if ($account->hasPermission('delete any affiliate_campaign entities')) {
          return AccessResult::allowed()
            ->cachePerPermissions()
            ->addCacheableDependency($entity);
        }

        // Owners can always delete their own campaigns (even if global).
        if ($this->checkOwn($entity, $operation, $account)) {
          return AccessResult::allowed()
            ->cachePerPermissions()
            ->cachePerUser()
            ->addCacheableDependency($entity);
        }

        return AccessResult::neutral()
          ->cachePerPermissions()
          ->addCacheableDependency($entity);
    }

    // Unknown operation, no opinion.
    return AccessResult::neutral()
      ->addCacheableDependency($entity);
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL) {
    return AccessResult::allowedIfHasPermission($account, 'create affiliate_campaign entities')
      ->cachePerPermissions();
  }

  /**
   * Test for given 'own' permission.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to check.
   * @param string $operation
   *   The operation being performed.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The account to check.
   *
   * @return bool
   *   TRUE if user has the own permission.
   */
  protected function checkOwn(EntityInterface $entity, $operation, AccountInterface $account) {
    $is_own = $account->isAuthenticated() && $account->id() == $entity->getOwnerId();
    if (!$is_own) {
      return FALSE;
    }

    $ops = [
      'view' => 'view own affiliate_campaign entities',
      'update' => 'edit own affiliate_campaign entities',
      'delete' => 'delete own affiliate_campaign entities',
    ];

    return $account->hasPermission($ops[$operation] ?? '');
  }

}
