<?php

namespace Drupal\affiliated\Entity\Handler;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Access controller for the AffiliateClick entity.
 */
class AffiliateClickAccessControlHandler extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    /** @var \Drupal\affiliated\Entity\AffiliateClickInterface $entity */

    switch ($operation) {
      case 'view':
        if ($account->hasPermission('view any affiliate_click entities')) {
          return AccessResult::allowed()
            ->cachePerPermissions()
            ->addCacheableDependency($entity);
        }

        if ($this->checkOwn($entity, $operation, $account)) {
          return AccessResult::allowed()
            ->cachePerPermissions()
            ->cachePerUser()
            ->addCacheableDependency($entity);
        }

        return AccessResult::neutral()
          ->cachePerPermissions()
          ->addCacheableDependency($entity);

      case 'delete':
        if ($account->hasPermission('delete any affiliate_click entities')) {
          return AccessResult::allowed()
            ->cachePerPermissions()
            ->addCacheableDependency($entity);
        }

        if ($this->checkOwn($entity, $operation, $account)) {
          return AccessResult::allowed()
            ->cachePerPermissions()
            ->cachePerUser()
            ->addCacheableDependency($entity);
        }

        return AccessResult::neutral()
          ->cachePerPermissions()
          ->addCacheableDependency($entity);
    }

    // Unknown operation, no opinion.
    return AccessResult::neutral()
      ->addCacheableDependency($entity);
  }

  /**
   * Test for given 'own' permission.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to check.
   * @param string $operation
   *   The operation being performed.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The account to check.
   *
   * @return bool
   *   TRUE if user has the own permission.
   */
  protected function checkOwn(EntityInterface $entity, $operation, AccountInterface $account) {
    $is_own = $account->isAuthenticated() && $account->id() == $entity->getOwnerId();
    if (!$is_own) {
      return FALSE;
    }

    $ops = [
      'view' => 'view own affiliate_click entities',
      'delete' => 'delete own affiliate_click entities',
    ];

    return $account->hasPermission($ops[$operation] ?? '');
  }

}
