<?php

namespace Drupal\affiliated\Event;

use Drupal\affiliated\Entity\AffiliateConversionInterface;
use Symfony\Contracts\EventDispatcher\Event;

/**
 * Event dispatched before a conversion is created.
 *
 * Subscribers can reject the conversion to prevent it from being saved.
 * This allows implementing modules to validate if a sale qualifies for
 * commission before the conversion is recorded.
 *
 * This event is dispatched AFTER all conversion data is populated (label,
 * commission, etc.) so subscribers have access to the complete conversion.
 *
 * When a subscriber calls $event->reject(), the custom storage handler
 * catches the exception and returns FALSE instead of the entity ID. The
 * rejection is automatically logged. Calling code can optionally check
 * the return value and get the rejection reason.
 *
 * Example subscriber:
 * @code
 * class MyConversionSubscriber implements EventSubscriberInterface {
 *
 *   public static function getSubscribedEvents(): array {
 *     return [
 *       ConversionPreCreateEvent::EVENT_NAME => ['onPreCreate'],
 *     ];
 *   }
 *
 *   public function onPreCreate(ConversionPreCreateEvent $event): void {
 *     $conversion = $event->getConversion();
 *     if ($this->shouldReject($conversion)) {
 *       $event->reject('Reason for rejection');
 *     }
 *   }
 *
 * }
 * @endcode
 *
 * Example checking for rejection (optional):
 * @code
 * if (!$conversion->save()) {
 *   // Conversion was rejected. Reason is already logged.
 *   // Optionally get the reason:
 *   $reason = $conversion->getRejectionReason();
 * }
 * @endcode
 *
 * @see \Drupal\affiliated\Storage\AffiliateConversionStorage
 * @see \Drupal\affiliated\Entity\AffiliateConversionInterface::getRejectionReason()
 */
class ConversionPreCreateEvent extends Event {

  /**
   * Event name constant.
   */
  const EVENT_NAME = 'affiliated_conversion_pre_create';

  /**
   * The conversion entity being created.
   *
   * @var \Drupal\affiliated\Entity\AffiliateConversionInterface
   */
  protected $conversion;

  /**
   * Whether the conversion has been rejected.
   *
   * @var bool
   */
  protected $rejected = FALSE;

  /**
   * The reason for rejection.
   *
   * @var string|null
   */
  protected $rejectionReason;

  /**
   * Constructs a ConversionPreCreateEvent.
   *
   * @param \Drupal\affiliated\Entity\AffiliateConversionInterface $conversion
   *   The conversion entity being created.
   */
  public function __construct(AffiliateConversionInterface $conversion) {
    $this->conversion = $conversion;
  }

  /**
   * Gets the conversion entity.
   *
   * @return \Drupal\affiliated\Entity\AffiliateConversionInterface
   *   The conversion entity.
   */
  public function getConversion(): AffiliateConversionInterface {
    return $this->conversion;
  }

  /**
   * Rejects the conversion, preventing it from being saved.
   *
   * @param string|null $reason
   *   Optional reason for the rejection.
   *
   * @return $this
   */
  public function reject(?string $reason = NULL): static {
    $this->rejected = TRUE;
    $this->rejectionReason = $reason;
    $this->stopPropagation();
    return $this;
  }

  /**
   * Checks if the conversion has been rejected.
   *
   * @return bool
   *   TRUE if rejected, FALSE otherwise.
   */
  public function isRejected(): bool {
    return $this->rejected;
  }

  /**
   * Gets the rejection reason.
   *
   * @return string|null
   *   The rejection reason, or NULL if not set.
   */
  public function getRejectionReason(): ?string {
    return $this->rejectionReason;
  }

}
