<?php

namespace Drupal\affiliated\Form;

use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides the default form handler for the Affiliate Campaign entity.
 */
class AffiliateCampaignForm extends ContentEntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state, $user = NULL): array {
    $form = parent::form($form, $form_state);
    $entity = $form_state->getFormObject()->getEntity();

    // Only allow users to change the affiliate and default status if they
    // can administer campaigns.
    if (!$this->currentUser()->hasPermission('administer affiliate_campaign entities')) {
      $form['user_id']['#access'] = FALSE;
      $form['is_default']['#access'] = FALSE;
    }

    // The is_global field requires the 'flag campaigns as global' permission.
    if (!$this->currentUser()->hasPermission('flag campaigns as global')) {
      $form['is_global']['#access'] = FALSE;
    }

    // If we are editing the default campaign, disable the checkbox.
    // You can only make it not the default by choosing another one as
    // the default.
    if ($entity->isDefault()) {
      $form['is_default']['#disabled'] = TRUE;
      // Default campaign must be global.
      $form['is_global']['#default_value'] = TRUE;
      $form['is_global']['#disabled'] = TRUE;
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $saved = parent::save($form, $form_state);
    $form_state->setRedirectUrl($this->entity->toUrl('canonical'));

    switch ($saved) {
      case SAVED_NEW:
        $this->messenger()->addMessage($this->t('Created campaign %label.', [
          '%label' => $this->entity->label(),
        ]));
        break;

      default:
        $this->messenger()
          ->addMessage($this->t('Updated the %label campaign.', [
            '%label' => $this->entity->label(),
          ]));
    }

    return $saved;
  }

}
