<?php

namespace Drupal\Tests\affiliated\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests the affiliate tracking endpoint.
 *
 * @group affiliated
 */
class AffiliateTrackingTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'field',
    'text',
    'options',
    'datetime',
    'views',
    'affiliated',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * An affiliate user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $affiliateUser;

  /**
   * The default campaign.
   *
   * @var \Drupal\affiliated\Entity\AffiliateCampaignInterface
   */
  protected $defaultCampaign;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Grant anonymous users 'access content' permission so the HTTP client
    // can access the tracking endpoint without a session.
    $this->container->get('entity_type.manager')
      ->getStorage('user_role')
      ->load('anonymous')
      ->grantPermission('access content')
      ->save();

    // Create an affiliate user.
    $this->affiliateUser = $this->drupalCreateUser(['act as an affiliate']);

    // Create a default campaign.
    $this->defaultCampaign = $this->container->get('entity_type.manager')
      ->getStorage('affiliate_campaign')
      ->create([
        'user_id' => 0,
        'name' => 'Default Campaign',
        'is_default' => 1,
        'is_global' => 1,
        'status' => 1,
      ]);
    $this->defaultCampaign->save();
  }

  /**
   * Tests that tracking endpoint requires affiliate parameter.
   */
  public function testTrackingRequiresAffiliate(): void {
    // Make a POST request without affiliate parameter.
    $client = $this->getHttpClient();
    $response = $client->post($this->buildUrl('/affiliated/track'), [
      'form_params' => [
        'landingPage' => '/products/test',
      ],
      'http_errors' => FALSE,
    ]);

    $statusCode = $response->getStatusCode();
    $contents = (string) $response->getBody();
    $body = json_decode($contents, TRUE);

    // Should return a 400 error for missing affiliate.
    $this->assertEquals(400, $statusCode, "Status: $statusCode, Body: $contents");
    $this->assertNotNull($body, "Response should be valid JSON. Status: $statusCode, Body: $contents");
    $this->assertFalse($body['tracked']);
    $this->assertEquals('Missing affiliate code', $body['message']);
  }

  /**
   * Tests tracking creates a click entity.
   */
  public function testTrackingCreatesClick(): void {
    // Make a POST request to the tracking endpoint.
    $client = $this->getHttpClient();
    $response = $client->post($this->buildUrl('/affiliated/track'), [
      'form_params' => [
        'affiliate' => $this->affiliateUser->id(),
        'landingPage' => '/products/test',
        'referrerUrl' => 'https://example.com',
      ],
      'http_errors' => FALSE,
    ]);

    $statusCode = $response->getStatusCode();
    $contents = (string) $response->getBody();
    $body = json_decode($contents, TRUE);

    $this->assertEquals(200, $statusCode, "Status: $statusCode, Body: $contents");
    $this->assertNotNull($body, "Response should be valid JSON. Status: $statusCode, Body: $contents");
    $this->assertTrue($body['tracked']);

    // Verify click was created.
    $clicks = $this->container->get('entity_type.manager')
      ->getStorage('affiliate_click')
      ->loadByProperties(['affiliate' => $this->affiliateUser->id()]);

    $this->assertCount(1, $clicks);
    $click = reset($clicks);
    $this->assertEquals('/products/test', $click->get('destination')->value);
  }

  /**
   * Tests tracking with campaign parameter.
   */
  public function testTrackingWithCampaign(): void {
    // Create a specific campaign.
    $campaign = $this->container->get('entity_type.manager')
      ->getStorage('affiliate_campaign')
      ->create([
        'user_id' => $this->affiliateUser->id(),
        'name' => 'Test Campaign',
        'status' => 1,
      ]);
    $campaign->save();

    $client = $this->getHttpClient();
    $response = $client->post($this->buildUrl('/affiliated/track'), [
      'form_params' => [
        'affiliate' => $this->affiliateUser->id(),
        'campaign' => $campaign->id(),
        'landingPage' => '/products/test',
        'referrerUrl' => 'https://example.com',
      ],
      'http_errors' => FALSE,
    ]);

    $statusCode = $response->getStatusCode();
    $contents = (string) $response->getBody();
    $body = json_decode($contents, TRUE);

    $this->assertEquals(200, $statusCode, "Status: $statusCode, Body: $contents");
    $this->assertNotNull($body, "Response should be valid JSON. Status: $statusCode, Body: $contents");
    $this->assertEquals($campaign->id(), $body['affiliate_campaign']);
  }

  /**
   * Tests tracking with invalid affiliate returns error.
   */
  public function testTrackingWithInvalidAffiliate(): void {
    $client = $this->getHttpClient();
    $response = $client->post($this->buildUrl('/affiliated/track'), [
      'form_params' => [
        'affiliate' => '999999',
        'landingPage' => '/products/test',
      ],
      'http_errors' => FALSE,
    ]);

    $statusCode = $response->getStatusCode();
    $contents = (string) $response->getBody();
    $body = json_decode($contents, TRUE);

    $this->assertEquals(200, $statusCode, "Status: $statusCode, Body: $contents");
    $this->assertNotNull($body, "Response should be valid JSON. Status: $statusCode, Body: $contents");
    $this->assertFalse($body['tracked']);
  }

  /**
   * Tests that non-affiliate users cannot be tracked.
   */
  public function testTrackingRejectsNonAffiliate(): void {
    $regularUser = $this->drupalCreateUser([]);

    $client = $this->getHttpClient();
    $response = $client->post($this->buildUrl('/affiliated/track'), [
      'form_params' => [
        'affiliate' => $regularUser->id(),
        'landingPage' => '/products/test',
      ],
      'http_errors' => FALSE,
    ]);

    $statusCode = $response->getStatusCode();
    $contents = (string) $response->getBody();
    $body = json_decode($contents, TRUE);

    $this->assertEquals(200, $statusCode, "Status: $statusCode, Body: $contents");
    $this->assertNotNull($body, "Response should be valid JSON. Status: $statusCode, Body: $contents");
    $this->assertFalse($body['tracked']);
  }

  /**
   * Tests click_precedence 'deny' mode - first click takes precedence.
   */
  public function testClickPrecedenceDenyKeepsFirstAffiliate(): void {
    // Set click_precedence to 'deny' (first click takes precedence).
    $this->config('affiliated.settings')->set('click_precedence', 'deny')->save();

    // Create a second affiliate user.
    $secondAffiliate = $this->drupalCreateUser(['act as an affiliate']);

    $client = $this->getHttpClient();

    // First visit - should track.
    $response1 = $client->post($this->buildUrl('/affiliated/track'), [
      'form_params' => [
        'affiliate' => $this->affiliateUser->id(),
        'campaign' => $this->defaultCampaign->id(),
        'landingPage' => '/products/first',
      ],
      'http_errors' => FALSE,
    ]);

    $contents1 = (string) $response1->getBody();
    $body1 = json_decode($contents1, TRUE);
    $this->assertNotNull($body1, "First response invalid. Status: {$response1->getStatusCode()}, Body: $contents1");
    $this->assertTrue($body1['tracked'], 'First visit should be tracked');
    $this->assertEquals($this->affiliateUser->id(), $body1['affiliate_id']);

    // Second visit with different affiliate - should NOT track (deny mode).
    $response2 = $client->post($this->buildUrl('/affiliated/track'), [
      'form_params' => [
        'affiliate' => $secondAffiliate->id(),
        'landingPage' => '/products/second',
      ],
      'headers' => [
        'Cookie' => 'affiliate_id=' . $this->affiliateUser->id() . '; affiliate_campaign=' . $this->defaultCampaign->id(),
      ],
      'http_errors' => FALSE,
    ]);

    $contents2 = (string) $response2->getBody();
    $body2 = json_decode($contents2, TRUE);
    $this->assertNotNull($body2, "Second response invalid. Status: {$response2->getStatusCode()}, Body: $contents2");
    $this->assertFalse($body2['tracked'], 'Second visit should NOT be tracked in deny mode');

    // Verify only 1 click was recorded (the first one).
    $clicks = $this->container->get('entity_type.manager')
      ->getStorage('affiliate_click')
      ->loadMultiple();
    $this->assertCount(1, $clicks, 'Only one click should be recorded in deny mode');
  }

  /**
   * Tests click_precedence 'overwrite' mode - new click takes precedence.
   */
  public function testClickPrecedenceOverwriteUpdatesAffiliate(): void {
    // Set click_precedence to 'overwrite' (new click takes precedence).
    $this->config('affiliated.settings')->set('click_precedence', 'overwrite')->save();

    // Create a second affiliate user.
    $secondAffiliate = $this->drupalCreateUser(['act as an affiliate']);

    $client = $this->getHttpClient();

    // First visit - should track.
    $response1 = $client->post($this->buildUrl('/affiliated/track'), [
      'form_params' => [
        'affiliate' => $this->affiliateUser->id(),
        'campaign' => $this->defaultCampaign->id(),
        'landingPage' => '/products/first',
      ],
      'http_errors' => FALSE,
    ]);

    $contents1 = (string) $response1->getBody();
    $body1 = json_decode($contents1, TRUE);
    $this->assertNotNull($body1, "First response invalid. Status: {$response1->getStatusCode()}, Body: $contents1");
    $this->assertTrue($body1['tracked'], 'First visit should be tracked');

    // Second visit with different affiliate - SHOULD track (overwrite mode).
    $response2 = $client->post($this->buildUrl('/affiliated/track'), [
      'form_params' => [
        'affiliate' => $secondAffiliate->id(),
        'campaign' => $this->defaultCampaign->id(),
        'landingPage' => '/products/second',
      ],
      'headers' => [
        'Cookie' => 'affiliate_id=' . $this->affiliateUser->id() . '; affiliate_campaign=' . $this->defaultCampaign->id(),
      ],
      'http_errors' => FALSE,
    ]);

    $contents2 = (string) $response2->getBody();
    $body2 = json_decode($contents2, TRUE);
    $this->assertNotNull($body2, "Second response invalid. Status: {$response2->getStatusCode()}, Body: $contents2");
    $this->assertTrue($body2['tracked'], 'Second visit SHOULD be tracked in overwrite mode');
    $this->assertEquals($secondAffiliate->id(), $body2['affiliate_id'], 'New affiliate should be recorded');

    // Verify 2 clicks were recorded.
    $clicks = $this->container->get('entity_type.manager')
      ->getStorage('affiliate_click')
      ->loadMultiple();
    $this->assertCount(2, $clicks, 'Two clicks should be recorded in overwrite mode');
  }

  /**
   * Tests that same affiliate with same campaign is not re-tracked.
   */
  public function testSameAffiliateNotRetracked(): void {
    // Set click_precedence to 'overwrite'.
    $this->config('affiliated.settings')->set('click_precedence', 'overwrite')->save();

    $client = $this->getHttpClient();

    // First visit.
    $response1 = $client->post($this->buildUrl('/affiliated/track'), [
      'form_params' => [
        'affiliate' => $this->affiliateUser->id(),
        'campaign' => $this->defaultCampaign->id(),
        'landingPage' => '/products/first',
      ],
      'http_errors' => FALSE,
    ]);

    $contents1 = (string) $response1->getBody();
    $body1 = json_decode($contents1, TRUE);
    $this->assertNotNull($body1, "First response invalid. Status: {$response1->getStatusCode()}, Body: $contents1");
    $this->assertTrue($body1['tracked'], 'First visit should be tracked');

    // Second visit with SAME affiliate and campaign - should NOT track.
    $response2 = $client->post($this->buildUrl('/affiliated/track'), [
      'form_params' => [
        'affiliate' => $this->affiliateUser->id(),
        'campaign' => $this->defaultCampaign->id(),
        'landingPage' => '/products/second',
      ],
      'headers' => [
        'Cookie' => 'affiliate_id=' . $this->affiliateUser->id() . '; affiliate_campaign=' . $this->defaultCampaign->id(),
      ],
      'http_errors' => FALSE,
    ]);

    $contents2 = (string) $response2->getBody();
    $body2 = json_decode($contents2, TRUE);
    $this->assertNotNull($body2, "Second response invalid. Status: {$response2->getStatusCode()}, Body: $contents2");
    $this->assertFalse($body2['tracked'], 'Same affiliate+campaign should not be re-tracked');

    // Verify only 1 click was recorded.
    $clicks = $this->container->get('entity_type.manager')
      ->getStorage('affiliate_click')
      ->loadMultiple();
    $this->assertCount(1, $clicks, 'Only one click should be recorded for same affiliate+campaign');
  }

  /**
   * Tests that different campaign with same affiliate IS tracked.
   *
   * When in overwrite mode, changing the campaign should create a new click.
   */
  public function testDifferentCampaignTrackedInOverwriteMode(): void {
    // Set click_precedence to 'overwrite'.
    $this->config('affiliated.settings')->set('click_precedence', 'overwrite')->save();

    // Create a second campaign.
    $secondCampaign = $this->container->get('entity_type.manager')
      ->getStorage('affiliate_campaign')
      ->create([
        'user_id' => $this->affiliateUser->id(),
        'name' => 'Second Campaign',
        'status' => 1,
      ]);
    $secondCampaign->save();

    $client = $this->getHttpClient();

    // First visit with default campaign.
    $response1 = $client->post($this->buildUrl('/affiliated/track'), [
      'form_params' => [
        'affiliate' => $this->affiliateUser->id(),
        'campaign' => $this->defaultCampaign->id(),
        'landingPage' => '/products/first',
      ],
      'http_errors' => FALSE,
    ]);

    $contents1 = (string) $response1->getBody();
    $body1 = json_decode($contents1, TRUE);
    $this->assertNotNull($body1, "First response invalid. Status: {$response1->getStatusCode()}, Body: $contents1");
    $this->assertTrue($body1['tracked'], 'First visit should be tracked');

    // Second visit with same affiliate but DIFFERENT campaign - SHOULD track.
    $response2 = $client->post($this->buildUrl('/affiliated/track'), [
      'form_params' => [
        'affiliate' => $this->affiliateUser->id(),
        'campaign' => $secondCampaign->id(),
        'landingPage' => '/products/second',
      ],
      'headers' => [
        'Cookie' => 'affiliate_id=' . $this->affiliateUser->id() . '; affiliate_campaign=' . $this->defaultCampaign->id(),
      ],
      'http_errors' => FALSE,
    ]);

    $contents2 = (string) $response2->getBody();
    $body2 = json_decode($contents2, TRUE);
    $this->assertNotNull($body2, "Second response invalid. Status: {$response2->getStatusCode()}, Body: $contents2");
    $this->assertTrue($body2['tracked'], 'Different campaign should be tracked in overwrite mode');
    $this->assertEquals($secondCampaign->id(), $body2['affiliate_campaign'], 'New campaign should be recorded');

    // Verify 2 clicks were recorded.
    $clicks = $this->container->get('entity_type.manager')
      ->getStorage('affiliate_click')
      ->loadMultiple();
    $this->assertCount(2, $clicks, 'Two clicks should be recorded for different campaigns');
  }

}
