<?php

namespace Drupal\Tests\affiliated\Kernel;

use Drupal\Core\Render\BubbleableMetadata;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;

/**
 * Tests affiliate conversion tokens.
 *
 * @group affiliated
 */
class TokenTest extends AffiliatedKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'views',
    'affiliated',
    'node',
    'field',
    'text',
    'filter',
  ];

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('node');
    $this->installConfig(['node', 'filter']);

    $this->token = $this->container->get('token');

    // Create a conversion type.
    $this->entityTypeManager->getStorage('affiliate_conversion_type')->create([
      'id' => 'test_conversion',
      'label' => 'Test Conversion',
      'description' => 'Test conversion type',
      'label_pattern' => '[affiliate_conversion:parent]',
    ])->save();

    // Create a node type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();
  }

  /**
   * Tests the parent token returns the parent entity label.
   */
  public function testParentToken(): void {
    $affiliate = $this->createAffiliateUser();

    // Create a node to use as parent.
    $node = Node::create([
      'type' => 'article',
      'title' => 'My Test Article',
      'uid' => $affiliate->id(),
    ]);
    $node->save();

    // Create a conversion with the node as parent.
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $affiliate->id(),
    ]);
    $conversion->setParentEntity($node);
    $conversion->save();

    // Test the parent token.
    $bubbleable_metadata = new BubbleableMetadata();
    $result = $this->token->replace(
      '[affiliate_conversion:parent]',
      ['affiliate_conversion' => $conversion],
      [],
      $bubbleable_metadata
    );

    $this->assertEquals('My Test Article', $result);
  }

  /**
   * Tests chained parent tokens.
   */
  public function testChainedParentToken(): void {
    $affiliate = $this->createAffiliateUser();

    // Create a node to use as parent.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Chained Token Test',
      'uid' => $affiliate->id(),
    ]);
    $node->save();

    // Create a conversion with the node as parent.
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $affiliate->id(),
    ]);
    $conversion->setParentEntity($node);
    $conversion->save();

    // Test chained token for node ID.
    $bubbleable_metadata = new BubbleableMetadata();
    $result = $this->token->replace(
      '[affiliate_conversion:parent:nid]',
      ['affiliate_conversion' => $conversion],
      [],
      $bubbleable_metadata
    );

    $this->assertEquals($node->id(), $result);
  }

  /**
   * Tests parent token with no parent entity.
   */
  public function testParentTokenWithoutParent(): void {
    $affiliate = $this->createAffiliateUser();

    // Create a conversion without a parent.
    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $affiliate->id(),
    ]);
    $conversion->save();

    // Token should not be replaced when no parent exists.
    $bubbleable_metadata = new BubbleableMetadata();
    $result = $this->token->replace(
      '[affiliate_conversion:parent]',
      ['affiliate_conversion' => $conversion],
      [],
      $bubbleable_metadata
    );

    // Token remains unreplaced when parent is NULL.
    $this->assertEquals('[affiliate_conversion:parent]', $result);
  }

  /**
   * Tests that parent token adds cache dependency.
   */
  public function testParentTokenCacheMetadata(): void {
    $affiliate = $this->createAffiliateUser();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Cache Test Article',
      'uid' => $affiliate->id(),
    ]);
    $node->save();

    $conversion = $this->entityTypeManager->getStorage('affiliate_conversion')->create([
      'type' => 'test_conversion',
      'affiliate' => $affiliate->id(),
    ]);
    $conversion->setParentEntity($node);
    $conversion->save();

    $bubbleable_metadata = new BubbleableMetadata();
    $this->token->replace(
      '[affiliate_conversion:parent]',
      ['affiliate_conversion' => $conversion],
      [],
      $bubbleable_metadata
    );

    // Verify the parent entity's cache tags were added.
    $this->assertContains('node:' . $node->id(), $bubbleable_metadata->getCacheTags());
  }

}
